unit main;

{$mode objfpc}{$H+}

{$include RELEASE_SETTINGS.INC}

{$IFDEF DEBUG_VERSION}
  // nothing jet here
{$ENDIF}

{*******************************************************************************
*                    Created by BREAKOUTBOX in 02-2011  as "PAwireEx"          *
*                    Rewritten and released in 2021 as "PortaudioDemo"         *
*                    Latest changes:  2025-12-05                               *
********************************************************************************
*                                                                              *
*  Such a component for a C DLL needs a lot of explanations, but in short,     *
*  here are some hints for usage :                                             *
*                                                                              *
*  - The application works now (2025-12-04) with both 32 and 64 bit.           *
*    There are a 32 bit and a 64 bit DLL in the exeOutput folder.              *
*                                                                              *
*  - BasicPortAudioIOobject.pas only contains the basic Portaudio IO code      *
*    and You shouldn't edit  BasicPortAudioIOobject.pas  at all                *
*    ( indeed, improvements are welcome .. ! )                                 *
*                                                                              *
*  - DemoPortaudioIOobject.pas is Your TEMPLATE unit, it contains              *
*    the TObject "TDemoPortAudioIO" that You can edit and play around with.    *
*                                                                              *
*  - both TBasicPortAudioIO and TDemoPortAudioIO are set to                    *
*    "PA_STREAMTYPE_STEREO" - this should work in all cases                    *
*    for Your first experience with Portaudio.dll ..                           *
*                                                                              *
*  - TBasicPortAudioIO is limited in some ways, but might suit Your needs (?)  *
*    - BUFFER_FRAMES is fixed value :  see TypesAndConst.pas                   *
*    - selecting Host, and IO Devices is (only) accessible via TComboBoxes     *
*      Therefor I implemented the "PortaudioAPIselector" GUI component         *
*    - TBasicPortAudioIO simply puts through the audio for demonstration.      *
*      Any DSP code You need, should be added in TDemoPortAudioIO              *
*      For sure You are free to rename "TDemoPortAudioIO" and / or             *
*      the corresponding unit "DemoPortaudioIOobject.pas" to Your needs ..     *
*                                                                              *
*                                                                              *
*                         +++ MOST IMPORTANT +++                               *
*                                                                              *
*    The Portaudio.dll Callback "hates" to be interrupted by for example       *
*    - GetMem() / FreeMem()                                                    *
*    - ANY GUI code You'd call inside the Callback                             *
*    - and in general :  DEBUGGING                                             *
*                                                                              *
*    For sure You should develope Your code with activated DEBUG, but ..       *
*    If You want Your application to properly run without any audio dropouts,  *
*    YOU MUST test it OUTSIDE the Lazarus IDE, and all DEBUG switched OFF !!!  *
*                                                                              *
*    To help You remind this, I implemented                                    *
*    - a debug / release include file called "RELEASE_SETTINGS.INC"            *
*      The topmost switch "[$define DEBUG_VERSION]" drives this                *
*    - and some helper code to remind You switching ON or OFF the Debuggger    *
*      => var DebugInfo in MainForm, and more ..                               *
*                                                                              *
*                                                                              *
*    And again, any improvements are welcome !                                 *
*                                                                              *
********************************************************************************
* 0.01  initial branch of Demo Project as "PortaudioDemo"                      *
* 0.02  included a different portaudio.pas file, inspired by UOS               *
* 0.03  removed dependency to main.pas in PortaudioAPIselector.pas             *
* 0.04  some minor changes                                                     *
* 0.05  cleanup in BasicPortAudioIOobject.pas                                  *
* 0.06  final "release", as a Portaudio example that works with INTEGER I/O    *
* 0.07  changes for making code run with both 32 and 64 bit Lazarus            *
*
* started adding both in and float usage
*******************************************************************************}

interface

uses
  Interfaces, LCLIntf, LMessages, Forms, Dialogs, Classes, Buttons, StdCtrls,
  ExtCtrls, SysUtils, Graphics, ComCtrls, FileUtil,

  Basicportaudioioobject,
  {$IFDEF USE_TDemoPortAudioIO}
  DemoPortaudioIOobject,
  {$ENDIF}
  PortaudioAPIselector,
  TypesAndConst;


{$IFDEF DEBUG_VERSION}
const VersionStr = '0.07 debug';
{$ELSE}
const VersionStr = '0.07';
{$ENDIF}


// put in main unit
var
  DebugInfo:Boolean = {$ifopt D+}true{$else}false{$endif};



const
  LM_AFTER_SHOW = LM_USER + 1000; // custom message


{ --- TForm1 ----------------------------------------------------------------- }
type
  { TForm1 }
  TForm1 = class(TForm)
    ButtonClose: TButton;
    ButtonSaveMemo: TButton;
    EditFileName: TEdit;
    ImageWaveForm: TImage;
    LibraryName: TLabel;
    LabelVolumeTxt: TLabel;
    LabelVolume: TLabel;
    LogfileMemo: TMemo;
    OpenDialog1: TOpenDialog;
    ProgressBar1: TProgressBar;
    StaticTextCpuLoad: TStaticText;
    StaticTextPATime: TStaticText;
    StaticTextPAFlags: TStaticText;
    TrackBarVolume: TTrackBar;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormClose(Sender: TObject; var {%H-}CloseAction: TCloseAction);
    // -----------------------------------------
    procedure ButtonCloseClick(Sender: TObject);
    procedure TrackBarVolumeChange(Sender: TObject);
  private
    { private declarations }
    PortaudioAPIselector : TPortaudioAPIselector;  // added 2021-03-08
    procedure Enable_EditAudioSettings( {%H-}b:boolean);
    procedure OnAppIdle(Sender: TObject; var Done: Boolean);
    procedure PaintWaveForm( pBuffer:POUTPUT_SAMPLE);
    // --- handle messages -----------------------------------------------------
    procedure WmAfterShow( var {%H-}Msg: TLMessage); message LM_AFTER_SHOW;   // added 2021-02-10
    procedure WmAPISelectorStatus( var Msg: TLMessage); message LM_PA_APISELECTOR;  // added 2021-03-09
    procedure WmPortaudioStatusFlags( var Msg: TLMessage); message LM_PA_STATUSFLAGS;  // added 2021-03-10
  protected
    { protected declarations }
  public
    { public declarations }
    Constructor Create(AOwner: TComponent); override;
    Destructor  Destroy; override;
  end;


var
  Form1: TForm1;

var
  {$IFDEF USE_TDemoPortAudioIO}
  PAObject : TDemoPortAudioIO;  // global VAR
  {$ELSE}
  PAObject : TBasicPortAudioIO;  // global VAR
  {$ENDIF}



implementation

{$R *.lfm}

uses
  LConvEncoding,
  {$IFDEF USE_TDemoPortAudioIO}
  math,
  {$ENDIF}
  portaudio,  // added 2025-12-02
  DebugIntoMemo;


{ ----- TForm1 --------------------------------------------------------------- }
constructor TForm1.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  // add some code here ..
end;


procedure TForm1.FormCreate(Sender: TObject);
begin
  {$IF Defined(MSWINDOWS)}
    {$IF Defined(WIN32)}
    Caption:= Caption +' 32 bit';
    {$ELSE}
    Caption:= Caption +' 64 bit';
    {$ENDIF}
  {$ELSEIF Defined(UNIX)}
    Caption:= Caption +' 64 bit';
  {$IFEND}

  {$IFDEF USE_TDemoPortAudioIO}
  Caption:= Caption +' - using TDemoPortAudioIO';
  {$ELSE}
  Caption:= Caption +' - using TBasicPortAudioIO';
  LabelVolume.Left:= TrackBarVolume.Left +10;
  LabelVolume.Caption:= 'no Volume Change possible';
  {$ENDIF}

  {$IFDEF PORTAUDIO_USE_FLOAT_IO}
  Caption:= Caption +' (float I/O)';
  {$ELSE}
  Caption:= Caption +' (integer I/O)';
  {$ENDIF}

  // added 2025-12-02 - put name of DLL or LIB onto Form !
  LibraryName.Caption:= portaudio.LibName;

  // from https://forum.lazarus.freepascal.org/index.php?topic=46696.0
  // detect DEBUG mode :
  {$ifopt D+}Caption:= Caption +'  [debug mode]';{$endif}


  // ---------------------------------------------
  // Create Your PortAudio Object
  //   ( With conditional switch "USE_TDemoPortAudioIO"
  //     You can switch back - on compile time -
  //     to TBasicPortAudioIO to see if things still work .. )
  {$IFDEF USE_TDemoPortAudioIO}
  PAObject:= TDemoPortAudioIO.Create( Self, @DemoPortaudioIOobject.PortAudioCallback);
  {$ELSE}
  PAObject:= TBasicPortAudioIO.Create( Self,
                                       @BasicPortAudioIOobject.PortAudioCallback,
                                       PA_STREAMTYPE_STEREO);
  {$ENDIF}

  Enable_EditAudioSettings( FALSE);

  // init vars:
  pGOutputBuffer:= NIL;

  // on Idle do painting :
  Application.AddOnIdleHandler( @OnAppIdle);
end;


procedure TForm1.FormShow(Sender: TObject);
begin
  {$IFDEF DEBUG_VERSION}
  {$IFOPT D-}{$NOTE debug mode is not active}{$ENDIF}  // user defined message
  if not DebugInfo
    then ShowMessage( 'Keep in mind - for a debug version You must switch on' +#13
                      +'"generate info for debugger" in Project settings !');
  {$ELSE}
  {$IFOPT D+}{$NOTE debug mode is active}{$ENDIF}      // user defined message
  if DebugInfo
    then ShowMessage( 'Keep in mind - for a release version You must switch off' +#13
                      +'"generate info for debugger" in Project settings !');
  {$ENDIF}


  // for DEBUG make TMemo visible everywhere via global var :
  pToLogfileMemo:= Self.LogfileMemo;

  // ---------------------------------------------
  // This object is the GUI for the PortAudio IO settings :
  PortaudioAPIselector:= TPortaudioAPIselector.Create( Self, Self, 0, 137,
                                                       TRUE, // dialog keeps visible ..
                                                       TBasicPortAudioIO( PAObject) );

  Self.LogfileMemo.Append( '...... #*+');

  // Select the Default API as text in the ComboBox
  // and assign device numbers to INPUT_DEVICE + OUTPUT_DEVICE :
  Memo_ShowPAVersionInformations( TBasicPortAudioIO( PAObject),
                                  LogfileMemo,
                                  portaudio.LibName);
  // ---------------------------------------------

  // force WmAfterShow() to be executed
  PostMessage( Self.Handle, LM_AFTER_SHOW, 0, 0);
end;


procedure TForm1.WmAfterShow( var Msg: TLMessage);
begin
  // do something after FormShow ..
end;


procedure TForm1.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
  // -----------------------------------------------
  // destroy the manually created PortAudio Object ?
  // => You've passed Your form as owner to this Object, so don't free manually !
  //PAObject.Free;
  // -----------------------------------------------
end;


destructor TForm1.Destroy;
begin
  // do something on Destroy ..

  Inherited Destroy;
end;


// -----------------------------------------------------------------------------
procedure TForm1.Enable_EditAudioSettings( b:boolean);
begin
  {$IFDEF USE_TDemoPortAudioIO}
  // do something on Edit Portaudio IO settings ..
  {$ENDIF}
end;


procedure TForm1.WmAPISelectorStatus( var {%H-}Msg: TLMessage); //message LM_PA_APISELECTOR;
begin
  case Msg.msg of
    // depending on TRUE / FALSE enable or disable :
    LM_PA_APISELECTOR:  Enable_EditAudioSettings( boolean( Msg.wParam));
  end;
end;


procedure TForm1.WmPortaudioStatusFlags( var Msg: TLMessage); //message LM_PA_STATUSFLAGS;  // added 2021-03-10
var
  s : string;
begin
  // http://www.portaudio.com/docs/proposals/001-UnderflowOverflowHandling.html
  // The following bits may be set in the statusFlags parameter.
  //
  // #define paInputUnderflow   ((PaStreamCallbackFlags)0x01)
  // #define paInputOverflow    ((PaStreamCallbackFlags)0x02)
  // #define paOutputUnderflow  ((PaStreamCallbackFlags)0x04)
  // #define paOutputOverflow   ((PaStreamCallbackFlags)0x08)
  // #define paPrimingOutput ((PaStreamCallbackFlags) 0x00000010)


  // DRAGONS - rework this - analyse all flags separately !
  case Msg.wParam of
     $01: s:= 'paInputUnderflow';
     $02: s:= 'paInputOverflow';
     $04: s:= 'paOutputUnderflow';
     $08: s:= 'paOutputOverflow';

     $05: s:= 'paInputUnderflow + paOutputUnderflow';
     $06: s:= 'paInputOverflow + paOutputUnderflow';
     $0C: s:= 'paOutputUnderflow + paOutputOverflow';
     else s:= 'unknown combination of flags'
  end;

  LogfileMemo.Append( 'LM_STATUSFLAGS' +#9 +'- '
                      +IntToHex( Msg.wParam, 2) +' ' +s);

  StaticTextPAFlags.Caption:= '$' +IntToHex( Msg.wParam, 8);
end;


procedure TForm1.OnAppIdle(Sender: TObject; var Done: Boolean);
{$IFDEF USE_TDemoPortAudioIO}
var
  maxValue : dword;
  pTemp    : POUTPUT_SAMPLE;
  i        : integer;
{$ENDIF}
begin
  Done:= TRUE;

  // on Idle we can paint the WaveForm and show some PortAudio values, like "CPU usage" ..
  if (pGOutputBuffer <> NIL) then
    begin
      if Assigned( PAObject) then
        begin
          {$IFDEF USE_TDemoPortAudioIO}
          Self.PaintWaveForm( pGOutputBuffer);

          // a primitive peak meter ..
          maxValue:= 0;
          pTemp:= pGOutputBuffer;
          for i:= 1 to IO_BUFFER_FRAMES *2 do
            begin
              {$ifdef PORTAUDIO_USE_FLOAT_IO}   // if activated, we use float32 instead of int16 !
              maxValue:= max( maxValue, Round( abs( pTemp^) *256 *256) );
              {$ELSE}
              maxValue:= max( maxValue, abs( pTemp^));
              {$ENDIF}
              inc( pTemp);
            end;

          // https://stackoverflow.com/questions/58832372/calculation-of-sound-level-in-db
          ProgressBar1.Position:=  96 +round( 20 * Log10( maxValue / 32767));
          {$ELSE}
          Self.PaintWaveForm( pGOutputBuffer);
          {$ENDIF}

          // only write if changed :
          if StaticTextCpuLoad.Caption <> PAObject.PA_CpuLoadStr
            then StaticTextCpuLoad.Caption:= 'CPU load:  ' +PAObject.PA_CpuLoadStr;
          if StaticTextPATime.Caption <> PAObject.PA_TimeStr
            then StaticTextPATime.Caption:= PAObject.PA_TimeStr;
          //StaticTextPAFlags.Caption := ;
        end;
    end;
end;


{ --- Buttons ---------------------------------------------------------------- }
procedure TForm1.ButtonCloseClick(Sender: TObject);
begin
  Close;
end;


{ --- TrackBar --------------------------------------------------------------- }
procedure TForm1.TrackBarVolumeChange(Sender: TObject);
begin
  {$IFDEF USE_TDemoPortAudioIO}
  PAObject.Volume:= TrackBarVolume.Position;
  LabelVolume.Caption:= IntToStr( TrackBarVolume.Position);
  {$ELSE}
  //
  {$ENDIF}
end;


{ --- functions -------------------------------------------------------------- }
procedure TForm1.PaintWaveForm( pBuffer:POUTPUT_SAMPLE);
var
  pTempINPUT : POUTPUT_SAMPLE;
  i          : Cardinal;
begin
  if (pBuffer <> NIL) then
  // fill a TImage with a WaveForm graphic
  with ImageWaveForm.Picture.Bitmap do
    begin
      pTempINPUT:= pBuffer;

      Width:= 512;
      Height:= Self.ImageWaveForm.Height -1;
      Canvas.Brush.Color:= $00000;
      Canvas.FillRect( 0, 0, Width, Height);

      Canvas.Pen.Color:= $00FF00;
      Canvas.MoveTo( 0, (Height div 2)
                         {$ifdef PORTAUDIO_USE_FLOAT_IO}   // if activated, we use float32 instead of int16 !
                         +Round (pTempINPUT^ * Height) );// div 256 div 256);
                         {$ELSE}
                         +pTempINPUT^ * Height div 256 div 256);
                         {$ENDIF}

      for i:= 1 to Width do
        begin
          Canvas.LineTo( i,
                         (Height div 2)
                         {$ifdef PORTAUDIO_USE_FLOAT_IO}   // if activated, we use float32 instead of int16 !
                         +Round (pTempINPUT^ * Height) );// div 256 div 256);
                         {$ELSE}
                         +pTempINPUT^ * Height div 256 div 256);
                         {$ENDIF}
          inc( pTempINPUT, 2);
        end;
    end;
end;


initialization


end.

