unit BasicPortAudioIOobject;

{$include RELEASE_SETTINGS.INC}

{$mode objfpc}{$H+}

{$IFDEF DEBUG_VERSION}
  {$DEFINE DEBUG_THIS_UNIT}
{$ENDIF}

{*******************************************************************************
*                    BasicPortAudioIOobject.pas                                *
*                                                                              *
*      Basic Object showing 32 bit interleaved stereo audio I/O                *
*                                                                              *
*                                                                              *
*  In fn Init_StreamParameters() this example application sets Portaudio I/O   *
*  to 32 bit FLOAT input / output = paFloat32  ( signed 32 bit )               *
*                                                                              *
*  You should not edit TBasicPortAudioIO but instead write Your own descendent *
*  Object - like in DemoPortaudioIOobject.pas - to add or overwrite anything.  *
*                                                                              *
*                                                                              *
*                    Created 2021-02-16 by BREAKOUTBOX                         *
*                    Latest changes:  2025-12-04                               *
*******************************************************************************}

interface

uses
  Interfaces,
  LCLIntf,    // for PostMessage() ..
  Classes, Controls, Forms, Dialogs, StdCtrls, ExtCtrls, SysUtils, CTypes,
  portaudio,
  TypesAndConst;



{ ---------------------------------------------------------------------------- }
{ Please take into account that Portaudio uses ( -10000 +x )  for errors       }
{ ---------------------------------------------------------------------------- }
const
  NO_ERROR      =  0;
  DEFAULT_ERROR = -1;


const
  // decides over the way You use PortAudio
  PA_STREAMTYPE_STEREO       = 0;
  PA_STREAMTYPE_MULTICHANNEL = 1;   // make use of all channels of a selected device


// This is the whole informations that Portaudio needs to be driven :
type
  PPortaudioConfig = ^TPortaudioConfig;
  TPortaudioConfig = record
          // record members used internally only:
          pCallbackFN            : TPaStreamCallback;
          pStream                : PPaStream;
          InputParameters        : TPaStreamParameters;
          OutputParameters       : TPaStreamParameters;

          // record members readable as property of TBasicPortAudioIO
          PA_IsRunning           : boolean;
          // PA config -------
          StreamType             : byte;     // see PA_open_and_start_Stream()
          TotalNumInputChannels  : integer;  // this is total num Channels available
          TotalNumOutputChannels : integer;  // this is total num Channels available
          SampleRate             : dword;        // our chosen "System Samplerate"
          FramesPerCallback      : integer;


          // device config ---
          PA_HOST_API            : integer;      // the chosen Host API
          HOST_API_NAME          : String;       // the name of the chosen Host API
          HOST_API_COUNT         : integer;      // number of avaliable Host APIs
          PA_INPUT_DEVICE        : integer;
          PA_OUTPUT_DEVICE       : integer;
          NUM_DEVICES            : integer;
          // device data -----
          SilentBuf              : array of byte;
          // information values --
          CpuLoad                : cdouble;
          PA_Time                : dword;
  end;


type
  TBasicPortAudioIO = class(TComponent)
  private
    { private declarations }
    function Init_paConfig_and_Buffers:Integer;
    function Init_Host_API:longint;//Integer;
    //function Update_Host_API( new_HOST_API:integer):Integer;
    function Update_Host_API( new_HOST_API:longint):longint;
    function Init_StreamParameters( aSampleRate:cint32):Integer;

    // property functions PortAudio -------------------------------------------
    procedure setOUTPUT_DEVICE( i:integer);
    procedure setINPUT_DEVICE( i:integer);
    function  getCpuLoadStr:string;
    function  getTimeStr:string;
  protected
    { protected declarations }
    paConfig            : TPortaudioConfig;  // user data record for callback
    fActiveStereoInput  : byte;
    fActiveStereoOutput : byte;  // added 2021-04-29

    procedure CallBack_ProcessBuffers( pInputBuffer, pOutputBuffer:pointer); //virtual; //not abstract;
  public
    { public declarations }
    //constructor Create(AOwner: TComponent; CallbackAdress:PPaStreamCallback;
    constructor Create(AOwner: TComponent; CallbackAdress:pointer;
                                           paStreamType: byte); reintroduce;
    destructor  Destroy; override;

    function PA_open_and_start_Stream:TPaError;
    function PA_stop_and_close_Stream:TPaError;

    function PA_Host_DeviceCount( aHOST_API:integer):integer;

    function PA_Device_NumInputChannels( aDeviceIndex:integer):integer;
    function PA_Device_NumOutputChannels( aDeviceIndex:integer):integer;
    function PA_Device_Index( aHOST_API, aDevice:integer):integer;

    // string functions --------------------------------------------------------
    function Str_PA_VersionNumber:string;
    function Str_PA_VersionText:string;
    // --------------------------------
    function Str_PA_HostAPICount:string;
    function Str_PA_HostAPIName( aHOST_API:integer):string;
    function Str_PA_HostAPI_Strings:string;
    function Str_PA_HostAPI_GetLastHostErrorText:string;
    function Str_PA_DeviceName( aDeviceIndex:integer):string;

    // ComboBox functions Host And Devices Selector ----------------------------
    procedure IOselector_init( var ComboBox1, ComboBox2, ComboBox3:TComboBox);
    procedure IOselector_ChangeHostAPI( ComboBox1, ComboBox2, ComboBox3:TComboBox; newHostAPIindex:TPaHostApiIndex);
    procedure IOselector_ChangeInputDevice( var ComboBoxInput:TComboBox);
    procedure IOselector_ChangeOutputDevice( var ComboBoxOutput:TComboBox);
  published
    { published declarations }

    // read only properties
    property PA_IsRunning:boolean read paConfig.PA_IsRunning;

    //property PA_DeviceNumInputChannels:integer read paConfig.DeviceNumInputChannels;
    //property PA_DeviceNumOutputChannels:integer read paConfig.DeviceNumOutputChannels;
    // 2021-04-29 changed to :
    property PA_NumInputChannels:integer read paConfig.InputParameters.channelCount;
    property PA_NumOutputChannels:integer read paConfig.OutputParameters.channelCount;
    property PA_SampleRate:dword read paConfig.SampleRate;
    property PA_FramesPerCallback:integer read paConfig.FramesPerCallback; // => BUFFER_FRAMES
    property PA_HostAPIcount:integer read paConfig.HOST_API_COUNT;   // COUNT depends on the DLL You use

    // simple vars
    property PA_StreamType:byte read paConfig.StreamType write paConfig.StreamType;
    property PA_HostAPI:integer read paConfig.PA_HOST_API write paConfig.PA_HOST_API;
    property PA_ActiveStereoInput:byte read fActiveStereoInput write fActiveStereoInput;
    property PA_ActiveStereoOutput:byte read fActiveStereoOutput write fActiveStereoOutput;


    // properties with function calls
    property PA_InputDevice:integer read paConfig.PA_INPUT_DEVICE write setINPUT_DEVICE;
    property PA_OutputDevice:integer read paConfig.PA_OUTPUT_DEVICE write setOUTPUT_DEVICE;

    // strings
    property PA_CpuLoadStr:string read getCpuLoadStr;
    property PA_TimeStr:string read getTimeStr;
  end;


var
  pGOutputBuffer : POUTPUT_SAMPLE = NIL;
                   // use as ReadOnly Buffer !  => paint WaveForm
                   //
                   // DRAGONS: currently this is a pointer to a
                   // stereo interleaved [cint16 = smallint] buffer (2024-07-14)
                   // .. but only in case the Audio Data
                   // are not floating point values  => POUTPUT_SAMPLE can be float !

{ This routine will be called by the PortAudio engine when audio is needed.
  It may be called at interrupt level on some machines so don't do anything
  that could mess up the system like calling malloc() or free().             }
function PortAudioCallback( pInputBuffer         : pointer;
                            pOutputBuffer        : pointer;
                            {%H-}FramesPerBuffer : culong;
                            pTimeInfo            : PPaStreamCallbackTimeInfo;
                            {%H-}PAStatusFlags   : TPaStreamCallbackFlags;
                            pUserData            : pointer) : CInt32; cdecl;


implementation

uses
  LConvEncoding,
  LazUTF8;


var
  iPortAudioCallback_DoNotEnter : boolean;

// This routine will be called by the PortAudio engine when audio is needed.
//   It may be called at interrupt level on some machines so don't do anything
//   that could mess up the system like calling malloc() or free().
function PortAudioCallback( pInputBuffer    : pointer;
                            pOutputBuffer   : pointer;
                            FramesPerBuffer : culong;
                            pTimeInfo       : PPaStreamCallbackTimeInfo;
                            PAStatusFlags   : TPaStreamCallbackFlags;
                            pUserData       : pointer) : CInt32; cdecl;
var
  pObject : TBasicPortAudioIO;
begin
  // The stream callback should return one of the values in the
  // PaStreamCallbackResult enumeration. To ensure that the callback continues
  // to be called, it should return paContinue (0)
  Result:= paContinue;

  // pUserData is = my TBasicPortAudioIO
  pObject:= TBasicPortAudioIO( pUserData);
  // save actual CPU usage / load of PortAudio.dll
  pObject.paConfig.CpuLoad:= PA_GetStreamCpuLoad( pObject.paConfig.pStream);
  pObject.paConfig.PA_Time:= Round( pTimeInfo^.CurrentTime);
                             // hint:  there are 3 Time values !


  //----------------------------------------------------------------------------
  // Protect against "Overload" - avoid accessing the code more than once :
  if iPortAudioCallback_DoNotEnter = TRUE then Exit;
  // DRAGONS - I don't know it this is ever required, but ..

  // on "FALSE" do DSP processing now ..
  iPortAudioCallback_DoNotEnter:= TRUE;

  // exit here if we don't run STEREO :    ( this is my design decision ! )
  // changed 2021-04-28
  //if (pObject.paConfig.DeviceNumInputChannels <> 2)
  //  or (pObject.paConfig.DeviceNumOutputChannels <> 2) then
  if (pObject.paConfig.InputParameters.channelCount <> 2)
    or (pObject.paConfig.OutputParameters.channelCount <> 2) then
    begin
      ShowMessage( 'DRAGONS - change this code - there''s no more STEREO !!!');

      Result:= -1;   // DRAGONS - error undefined
      Exit;
    end;


  // ---------------------------------------------------------------------------
  // HERE we "link in" the AudioDSP processing
  if Assigned( pObject)
    then pObject.CallBack_ProcessBuffers( pInputBuffer, pOutputBuffer);
  // ---------------------------------------------------------------------------


  // copy actual buffer pointer to a global pointer, for VU Meter etc. ( Readonly !) :
  pGOutputBuffer:= @pOutputBuffer^;
  // DRAGONS:  better COPY the buffer content to a global tempBuffer,
  //           pointer "pOutputBuffer" could become invalid .. !

  iPortAudioCallback_DoNotEnter:= FALSE;
end;



// to keep things simpler, we process Audio in this separate procedure
procedure TBasicPortAudioIO.CallBack_ProcessBuffers( pInputBuffer,
                                                     pOutputBuffer:pointer);
var
  pINPUT  : PINPUT_SAMPLE;
  pOUTPUT : POUTPUT_SAMPLE;
  i, n    : Dword;
begin
  // --- set Input Pointer ---
  // PA CallBack Function may get called with NULL inputBuffer,
  // during initial setup, or because there IS no input device !
  if pInputBuffer = NIL
     then pINPUT:= @paConfig.SilentBuf[0]   // read a dummy buffer
     else pINPUT:= pInputBuffer;

  // --- set Output Pointer ---
  pOUTPUT:= pOutputBuffer;


  // ---------------------------------------------------------------------------
  // Usually You would insert Your code here to insert Audio from File or do DSP ..
  // ... but ...
  //
  // Keep  TBasicPortAudioIO  as it is = a template for basic Portaudio IO.
  // Overwrite everything in Your own  IO object, derived from TBasicPortAudioIO !
  // ---------------------------------------------------------------------------


  // --- pointers are set, now copy from IN to OUT buffer ----------------------
  n:= paConfig.FramesPerCallback * 2;  // paConfig.PA_Frames_Per_Callback should be = BUFFER_FRAMES
  for i:= 1 to n do
    begin
      // copy both channels' audio data to Portaudio's output buffer
      pOUTPUT^:= pINPUT^;
      inc( pINPUT);
      inc( pOUTPUT);
    end;
end;


// -----------------------------------------------------------------------------
// ----------------- Create ----------------------------------------------------
// -----------------------------------------------------------------------------
constructor TBasicPortAudioIO.Create( AOwner: TComponent;
                                      CallbackAdress:pointer;
                                      paStreamType: byte);
var
  err : TPaError;
begin
  inherited Create(AOwner);

  // assign the local Callback Function address to a VAR in  paConfig :
  paConfig.pCallbackFN:= TPaStreamCallback( CallbackAdress);  // added 2021-03-07

  // init StreamType
  PA_StreamType:= paStreamType;   // see PA_open_and_start_Stream()


  {$ifdef LOAD_PA_ON_RUNTIME}
  if Pa_Load( LibName) = FALSE
    then ShowMessage( 'Error: Pa_Load( LibName) = FALSE');
  {$ENDIF}

  // --- init PA for Playback / Record / callback function ---------------------
  TRY
    err:= Pa_Initialize();
  except
    ShowMessage( 'Exception while Pa_Initialize()');
  end;
  if err <> paNoError
  {$IFDEF DEBUG_THIS_UNIT}
    then ShowMessage( 'Error while Pa_Initialize() in TCustomPortAudioObject.Create');
  {$ELSE}
    then ShowMessage( 'Error while Pa_Initialize()');
  {$ENDIF}


  //* init settings for startup : *//
  iPortAudioCallback_DoNotEnter := FALSE;
  paConfig.PA_IsRunning         := FALSE;

  // set up buffers and init SampleRate and so on ..
  Self.Init_paConfig_and_Buffers;

  // set default interface and devices
  Self.Init_Host_API;

  // set Stream I/O Parameters
  Self.Init_StreamParameters( 44100);  // fixed value by design, as I wanted it to be ..

  // open the Portaudio Stream
  err:= Self.PA_open_and_start_Stream;
  if err <> 0
    then
      begin
        ShowMessage( 'Error while PA_open_and_start_Stream() = ' +IntToStr( err));
      end;
end;


// ---------------- Destroy ----------------------------------------------------
Destructor TBasicPortAudioIO.Destroy;
var
  err : TPaError;
begin
  //* close the Portaudio Stream *//
  if PA_IsRunning then // added this 2024-08-12 ..
    begin
      err:= Self.PA_stop_and_close_Stream;
      if err <> NO_ERROR
      {$IFDEF DEBUG_THIS_UNIT}
        then ShowMessage( 'Error while PA_stop_and_close_Stream() in TBasicPortAudioIO.Destroy');
      {$ELSE}
        then ShowMessage( 'Error while PA_stop_and_close_Stream()');
      {$ENDIF}
    end;


  // Pa_Terminate() MUST be called before exiting a program which uses PortAudio.
  // Failure to do so may result in serious resource leaks, such as audio devices
  // not being available until the next reboot.
  err:= Pa_Terminate();
  if err <> paNoError
  {$IFDEF DEBUG_THIS_UNIT}
    then ShowMessage( 'Error while Pa_Terminate() in TBasicPortAudioIO.Destroy');
  {$ELSE}
    then ShowMessage( 'Error while Pa_Terminate()');
  {$ENDIF}


  {$ifdef LOAD_PA_ON_RUNTIME}
  Pa_Unload();
  {$ENDIF}

  Inherited Destroy;
end;


// ----------------- Init_Host_API ---------------------------------------------
// --- these functions require PortAudio to be initialized ---------------------
function TBasicPortAudioIO.Init_Host_API():longint;
begin
  // set DefaultHostApi as the actual API
  Result:= Update_Host_API( Pa_GetDefaultHostApi());
end;


function TBasicPortAudioIO.Init_paConfig_and_Buffers():Integer;
begin
  Result:= 0;

  with paConfig do
    begin
      // init Host API vars - as there where no interfaces and no IOs at all :
      PA_HOST_API      := -1;
      HOST_API_COUNT   := -1;
      PA_INPUT_DEVICE  := -1;
      PA_OUTPUT_DEVICE := -1;

      // num Channels
      case paConfig.StreamType of
        PA_STREAMTYPE_STEREO :
          begin
            TotalNumInputChannels  := 2;  // "stereo"
            TotalNumOutputChannels := 2;  // "stereo"
          end;
        PA_STREAMTYPE_MULTICHANNEL:
          begin
            // DRAGONS - what to do ?
          end;
      end;

      // set STATIC number of Buffer Frames - You could make it variable if You want ..
      paConfig.FramesPerCallback:= IO_BUFFER_FRAMES;

      { *** set up the SilentBuf :  *** }
      SetLength( SilentBuf, paConfig.FramesPerCallback
                            * SizeOf( INPUT_SAMPLE)   // 16 bit int or 32 bit float
                             * 2);  //* config.numInputChannels;

      { *** clean the SilentBuf :  *** }
      FillChar( SilentBuf[0], paConfig.FramesPerCallback
                              * SizeOf( INPUT_SAMPLE) // 16 bit
                              * 2    //* config.numInputChannels
                              , 0);  // fill with silence ..
    end;
end;


function TBasicPortAudioIO.Update_Host_API( new_HOST_API:integer):Integer;
var
  paApiInfo   : PPaHostApiInfo;
  DeviceIndex : cint;    // = TPaDeviceIndex in PortAudio.pas
  i           : integer;
  n           : dword;
begin
  Result:= 0;

  paConfig.PA_HOST_API:= new_HOST_API;

  // get Audio API count
  //paConfig.HOST_API_COUNT:= PTRUINT( Pa_GetHostApiCount);  // I expect it's always > 0  ..
  paConfig.HOST_API_COUNT:= Pa_GetHostApiCount();  // I expect it's always > 0  ..
  //ShowMessage( 'Debug:'
  //             +'  paConfig.PA_HOST_API = ' +IntToStr( paConfig.PA_HOST_API)
  //             +'  paConfig.HOST_API_COUNT = ' +IntToStr( paConfig.HOST_API_COUNT));

  // --- set input and output for first start ----------------------------------
  paApiInfo:= Pa_GetHostApiInfo( paConfig.PA_HOST_API);
  if paApiInfo = NIL then ShowMessage( 'Error:  paApiInfo = NIL');
  paConfig.HOST_API_NAME := paApiInfo^.name;
  paConfig.NUM_DEVICES   := paAPIinfo^.deviceCount;

  if paConfig.NUM_DEVICES > 0
    then
      begin
        // prevent error in case there's not even 1 available input device :
        paConfig.PA_INPUT_DEVICE:= paNoDevice;   // added 2021-02-28

        // set input
        for i:= 0 to paConfig.NUM_DEVICES -1 do
          begin
            DeviceIndex:= PA_Device_Index( paConfig.PA_HOST_API, i);
            // code changed 2021-04-28 ..
            n:= Self.PA_Device_NumInputChannels( DeviceIndex);
            if n > 0 then
              begin
                paConfig.PA_INPUT_DEVICE:= DeviceIndex;
                //paConfig.DeviceNumInputChannels:= n;//Self.PA_Device_NumInputChannels( DeviceIndex);

                //ShowMessage( 'TBasicPortAudioIO.Update_Host_API()' +#13#13
                //             //+'paConfig.DeviceNumInputChannels = '
                //             //+IntToStr( paConfig.DeviceNumInputChannels) +#13#13
                //             +'paConfig.InputParameters.channelCount = '
                //             +IntToStr( paConfig.InputParameters.channelCount) );
                Break;
              end;
          end;

        // set output
        for i:= 0 to paConfig.NUM_DEVICES -1 do
          begin
            DeviceIndex:= PA_Device_Index( paConfig.PA_HOST_API, i);
            if Self.PA_Device_NumOutputChannels( DeviceIndex) > 0 then
              begin
                paConfig.PA_OUTPUT_DEVICE:= DeviceIndex;
                paConfig.TotalNumOutputChannels:= Self.PA_Device_NumOutputChannels( DeviceIndex);
                Break;
              end;
          end;
      end
    else Result:= -1;
end;


function TBasicPortAudioIO.Init_StreamParameters( aSampleRate:cint32):Integer;
begin
  Result:= 0;

  with paConfig do
    begin
      { *** SampleRate *** }
      SampleRate:= aSampleRate;  //usually 44100

      { *** set I/O Default Devices *** }
      InputParameters.device  := Pa_GetDefaultInputDevice();
      // WARNING:  input probably is  = -1  =  paNoDevice !!!
      OutputParameters.device := Pa_GetDefaultOutputDevice();

      { *** request 2 Channels from Stream I/O *** }
      InputParameters.channelCount  := 2;  // = stereo
      OutputParameters.channelCount := 2;  // = stereo

      { *** set Sample Format *** }
      //  The paNonInterleaved flag ($80000000) indicates that a
      // multichannel buffer is passed as a set of non-interleaved pointers.
      // =>
      // In  TBasicPortAudioIO  currently "Interleaved" is used for Audio I/O
      //
      {$ifdef PORTAUDIO_USE_FLOAT_IO}
      // new 2021-04-17 - if activated, we use float32 instead of int16 !
      InputParameters.sampleFormat  := paFloat32;  // signed 32 bit float
      OutputParameters.sampleFormat := paFloat32;  // signed 32 bit float
      //  The floating point representation (paFloat32) uses +1.0 and -1.0
      //  as the maximum and minimum respectively.
      {$ELSE}             // if not, we use int16 ..
      InputParameters.sampleFormat  := paInt16;  // signed 16 bit int
      OutputParameters.sampleFormat := paInt16;  // signed 16 bit int
      {$ENDIF}


      { *** suggested Latency *** }
      if InputParameters.device <> paNoDevice
        then InputParameters.suggestedLatency  := Pa_GetDeviceInfo( InputParameters.device)^.defaultLowInputLatency;
      if OutputParameters.device <> paNoDevice
        then OutputParameters.suggestedLatency := Pa_GetDeviceInfo( OutputParameters.device)^.defaultLowInputLatency;

      InputParameters.hostApiSpecificStreamInfo  := NIL;
      OutputParameters.hostApiSpecificStreamInfo := NIL;

      { *** set num Frames in buffer *** }
      paConfig.FramesPerCallback:= IO_BUFFER_FRAMES;   // see TypesAndConst.pas
    end;
end;


function TBasicPortAudioIO.PA_open_and_start_Stream:TPaError;
var
  pInputParameters : PPaStreamParameters;
  err : TPaError;
label error;
begin
  Result:= 0;
  err:= paNoError;

  // --- set parameters for  InputParameters  and  OutputParameters  -----

  { *** set ACTUAL input device properties : *** }
  paConfig.InputParameters.device:= paConfig.PA_INPUT_DEVICE;
  if paConfig.InputParameters.device = paNoDevice
    then
      begin
        err:= -1;
        // This is not really an error !
        // Maybe there simply IS no available INPUT !
        // => so here DO NOT DO:  goto error;
        pInputParameters:= NIL;  // => no device found, so set to NIL;
      end

    else
      begin
        paConfig.TotalNumInputChannels:=
          Pa_GetDeviceInfo( paConfig.InputParameters.device)^.maxInputChannels;

        case PA_StreamType of
            PA_STREAMTYPE_STEREO :
              begin
                // open only 2 of - probably - more than 2 Input Channels :
                paConfig.InputParameters.ChannelCount:= 2;
              end;
            PA_STREAMTYPE_MULTICHANNEL :
              begin
                // new 2021-03-21 - "open ALL inputs"
                paConfig.InputParameters.ChannelCount:= paConfig.TotalNumInputChannels;
              end;
        end;

        paConfig.InputParameters.suggestedLatency:=
          Pa_GetDeviceInfo( paConfig.InputParameters.device)^.defaultLowInputLatency;
        pInputParameters:= @paConfig.InputParameters;
      end;


  { *** set ACTUAL output device properties : *** }
  paConfig.OutputParameters.device:= paConfig.PA_OUTPUT_DEVICE;
  if paConfig.OutputParameters.device = paNoDevice
    then
      begin
        err:= -2;
        //Memo1.Append( 'Error: No (default) output device available.');
        ShowMessage( 'Error: No (default) output device available.');
        goto error;   // no Audio Output .. I don't like that ...
      end
    else
      begin
        paConfig.TotalNumOutputChannels:=
          Pa_GetDeviceInfo( paConfig.OutputParameters.device)^.maxOutputChannels;

        case PA_StreamType of
            PA_STREAMTYPE_STEREO :
              begin
                // open only 2 of - probably - more than 2 Input Channels :
                paConfig.OutputParameters.ChannelCount:= 2;
              end;
            PA_STREAMTYPE_MULTICHANNEL :
              begin
                //paConfig.OutputParameters.ChannelCount:= 2;
                // new 2021-04-29 - "open ALL inputs"
                paConfig.OutputParameters.ChannelCount:= paConfig.TotalNumOutputChannels;
              end;
        end;

        paConfig.OutputParameters.suggestedLatency:=
          Pa_GetDeviceInfo( paConfig.OutputParameters.device)^.defaultLowOutputLatency;
      end;

  {$IFDEF DEBUG_THIS_UNIT}
  //with paConfig do
  //  ShowMessage( 'TotalNumInputChannels = ' +IntToStr( TotalNumInputChannels) +#13
  //               +'TotalNumOutputChannels = ' +IntToStr( TotalNumOutputChannels) +#13#13
  //               +'InputParameters.ChannelCount = ' +IntToStr( InputParameters.ChannelCount) +#13
  //               +'OutputParameters.ChannelCount = ' +IntToStr( OutputParameters.ChannelCount) );
  {$ENDIF}

  // --- check on valid format, with 44.100 Hz ---------------------------------
  paConfig.SampleRate:= 44100;

  if paConfig.InputParameters.device = paNoDevice
    then err:= Pa_IsFormatSupported( pInputParameters,  // might be NIL
                                     @paConfig.OutputParameters,
                                     paConfig.SampleRate)
    else err:= Pa_IsFormatSupported( @paConfig.InputParameters,
                                     @paConfig.OutputParameters,
                                     paConfig.SampleRate);

  // in case 44.100 Hz did'n work, try try again, with 48.000 Hz ---------------
  if err <> paNoError then
    begin
      ShowMessage( 'Error with ' +IntToStr( paConfig.SampleRate) +' in PortAudioOpenStream ' +IntToStr( err));

      paConfig.SampleRate:= 48000;
      if paConfig.InputParameters.device = paNoDevice
        then err:= Pa_IsFormatSupported( pInputParameters,  // might be NIL
                                         @paConfig.OutputParameters,
                                         paConfig.SampleRate)
        else err:= Pa_IsFormatSupported( @paConfig.InputParameters,
                                         @paConfig.OutputParameters,
                                         paConfig.SampleRate);
      if err <> paNoError then
        begin
          ShowMessage( 'Error with ' +IntToStr( paConfig.SampleRate) +' in PortAudioOpenStream ' +IntToStr( err));
          // on failure, reset to 44.100 Hz
          paConfig.SampleRate:= 44100;
          goto error;
        end;
    end;

  // open stream
  err:= Pa_OpenStream( paConfig.pStream,
                       pInputParameters,  // might be NIL if "NoDevice"
                       @paConfig.OutputParameters,
                       paConfig.SampleRate,
                       paConfig.FramesPerCallback,  //** frames per buffer
                       //paClipOff,   //** WARNING - with [paClipOff]
                                      //** we output "out of range" samples !!!
                       paNoFlag,      // 2021-05-14 - changed to paNoFlag
                       PPaStreamCallback( paConfig.pCallbackFN),
                       // ---------------------------------------------------------------
                       Self);  // HERE we put the whole TBasicPortAudioIO into Pa_OpenStream()
                               // to be able to access all it's elements in the Callback
                               // ---------------------------------------------------------------
  if err <> paNoError then goto error;

  // start stream
  Err:= Pa_StartStream( paConfig.pStream);
  if Err = paNoError
    then paConfig.PA_IsRunning:= TRUE
    else goto error;

  // exit here if everything's OK :
  Exit;

error:
  Result:= err;
end;


function TBasicPortAudioIO.PA_stop_and_close_Stream:TPaError;
begin
  paConfig.PA_IsRunning:= FALSE;  // too early ???

  // stop the stream
  Result:= Pa_StopStream( paConfig.pStream);
  if Result = paNoError
    then
      begin
        //Memo1.Append( 'Portaudio Stream stopped successfully');
      end
    else ShowMessage( 'Pa_StopStream - Error message:'+#9 +Pa_GetErrorText( Result ));

  // Bugfix added 2021-04-17 - because TForm1.PaintWaveForm() crashed !!!
  pGOutputBuffer:= NIL;

  // close the stream
  Result:= Pa_CloseStream( paConfig.pStream);
  if Result <> paNoError
    then ShowMessage( 'Pa_CloseStream - Error message:'+#9 +Pa_GetErrorText( Result ))
    ;//else Memo1.Append( 'Portaudio Stream closed successfully');
end;


// -----------------------------------------------------------------------------
// --- String Functions --------------------------------------------------------
// -----------------------------------------------------------------------------
function TBasicPortAudioIO.Str_PA_VersionNumber:String;
begin
  Result:= 'PA version int:   ' +IntToStr( Pa_GetVersion() );

  // attention:  the following code compiles, but is wrong with "late binding" !
  //Result:= 'PA version int:   ' +IntToStr( cint( Pa_GetVersion));
end;


function TBasicPortAudioIO.Str_PA_VersionText:String;
begin
  // You should not be casting C string pointers directly to pascal string types
  // (which are reference counted amongst other things).
  //  For this you should use PChar then use StrPas to convert it to a(n) (ansi)string.

  Result:= 'PA version text:   ' +LineEnding +StrPas( Pa_GetVersionText() );
end;


function TBasicPortAudioIO.Str_PA_HostAPICount:String;
begin
  Result:= 'Host API count (ok if positive): ' +IntToStr( paConfig.HOST_API_COUNT);
end;


function TBasicPortAudioIO.Str_PA_HostAPIName( aHOST_API:integer):String;
var
  HaInfo : PPaHostApiInfo;
begin
  HaInfo:= Pa_GetHostApiInfo ( aHOST_API);
  if HaInfo = NIL
    then Result:= ''
    else Result:= WinCPToUTF8( HaInfo^.name);
end;


// added 2025-12-03 - inspired from https://forum.lazarus.freepascal.org/index.php/topic,44294.msg311677.html#msg311677
function DetectTwoByteChars( const AString: String):boolean;
var
  Codepoint : PChar = #0;
  Size      : Integer;
begin
  result:= false;

  if AString <> '' then
    begin
      Codepoint:= @AString[1];
      while Codepoint^ <> #0 do
      begin
        Size:= UTF8CodepointSize( Codepoint);
        Codepoint += Size;
        if Size > 1 then
          begin
            result:= true;
            break;
          end;
      end;
    end;
end;

function TBasicPortAudioIO.Str_PA_DeviceName( aDeviceIndex:integer):String;
var
  pDeviceInfo : PPaDeviceInfo;
begin
  pDeviceInfo:= Pa_GetDeviceInfo( aDeviceIndex);

  if pDeviceInfo = NIL
    then Result:= ''
    else
      begin
        // 2025-12-03 - new solution for "Encoding Problem":
        //
        // we need to convert the strings IF they're not "lazarus / Pascal" strings !
        // DRAGONS: sometimes the string already is UTF8 !
        if DetectTwoByteChars( pDeviceInfo^.name)
          then Result:= pDeviceInfo^.name
          //else Result:= CP1252ToUTF8( pDeviceInfo^.name);
          else Result:= WinCPToUTF8( pDeviceInfo^.name);

        {ShowMessage( 'DEBUG' +#13
                     +pDeviceInfo^.name +#13
                     +CP1252ToUTF8( pDeviceInfo^.name) +#13
                     +WinCPToUTF8( pDeviceInfo^.name));
        }
      end;
end;


function TBasicPortAudioIO.Str_PA_HostAPI_Strings:String;
var
  HaInfo : PPaHostApiInfo;
  i      : integer;
begin
  Result:= '';

  for i:= 0 to paConfig.HOST_API_COUNT -1 do
    begin
      HaInfo:= Pa_GetHostApiInfo ( i);
      Result:= Result +( '  Host API ' +IntToStr( i) +':  '
                      +IntToStr( HaInfo^.deviceCount)
                      +' devices ' +WinCPToUTF8( HaInfo^.name));  // WinCPToUTF8() ?
      if i < paConfig.HOST_API_COUNT -1
         then Result:= Result +LineEnding;
    end;
end;


function TBasicPortAudioIO.Str_PA_HostAPI_GetLastHostErrorText:String;
var
  pInfo : PPaHostErrorInfo;
begin
  // The values in this structure will only be valid if a PortAudio function
  // has previously returned the paUnanticipatedHostError error code.
  pInfo:= pointer( Pa_GetLastHostErrorInfo);

  Result:= pInfo^.errorText;
end;


// -----------------------------------------------------------------------------
// --- Integer Functions -------------------------------------------------------
// -----------------------------------------------------------------------------
function TBasicPortAudioIO.PA_Host_DeviceCount( aHOST_API:integer):integer;
var
  paApiInfo : PPaHostApiInfo;
begin
  paApiInfo:= Pa_GetHostApiInfo( aHOST_API);

  if paApiInfo = NIL
    then Result:= DEFAULT_ERROR
    else Result:= paAPIinfo^.deviceCount;
end;


function TBasicPortAudioIO.PA_Device_Index( aHOST_API, aDevice:integer):integer;
begin
  Result:= Pa_HostApiDeviceIndexToDeviceIndex( aHOST_API, aDevice);
end;


function TBasicPortAudioIO.PA_Device_NumInputChannels( aDeviceIndex:integer):integer;
var
  pDeviceInfo : PPaDeviceInfo;
begin
  pDeviceInfo:= Pa_GetDeviceInfo( aDeviceIndex);

  if pDeviceInfo = NIL
    then Result:= 0
    else Result:= pDeviceInfo^.maxInputChannels;
end;


function TBasicPortAudioIO.PA_Device_NumOutputChannels( aDeviceIndex:integer):integer;
var
  pDeviceInfo : PPaDeviceInfo;
begin
  pDeviceInfo:= Pa_GetDeviceInfo( aDeviceIndex);

  if pDeviceInfo = NIL
    then Result:= 0
    else Result:= pDeviceInfo^.maxOutputChannels;
end;


// -----------------------------------------------------------------------------
// Interface functions
// -----------------------------------------------------------------------------
procedure TBasicPortAudioIO.setINPUT_DEVICE( i:integer);
begin
  //if i < Pa_GetDeviceCount() then   // COULD be done, to be sure ..
    Self.paConfig.PA_INPUT_DEVICE:= i;
end;


procedure TBasicPortAudioIO.setOUTPUT_DEVICE( i:integer);
begin
  //if i < Pa_GetDeviceCount() then   // COULD be done, to be sure ..
    Self.paConfig.PA_OUTPUT_DEVICE:= i;
end;


// -----------------------------------------------------------------------------
// Interface functions read only
// -----------------------------------------------------------------------------
function TBasicPortAudioIO.getCpuLoadStr:string;
begin
  Result:= IntToStr( Round( 1000 * paConfig.CpuLoad) div 10)
                     +','
                     +IntToStr( Round( 1000 * paConfig.CpuLoad) mod 10)
                     +' %';
end;


function PA_TimeToStr( PATime:dword):string;
var
  minStr : string;
  secStr : string;
begin
  minStr:= IntToStr( PATime div 60 mod 60);
  secStr:= IntToStr( PATime mod 60);
  if Length( minStr) = 1 then minStr:= '0' +minStr;
  if Length( secStr) = 1 then secStr:= '0' +secStr;

  Result:= IntToStr( PATime div 3600) +':' +minStr +':' +secStr;
end;


function TBasicPortAudioIO.getTimeStr:string;
begin
  Result:= PA_TimeToStr( Round( Self.paConfig.PA_Time));
end;


// -----------------------------------------------------------------------------
// --- ComboBox functions Host And Devices Selector ----------------------------
// -----------------------------------------------------------------------------
procedure TBasicPortAudioIO.IOselector_init( var ComboBox1, ComboBox2,
                                                 ComboBox3 : TComboBox);
{$IF Defined(MSWINDOWS)}
  {$IF Defined(WIN32)}
  {$ELSE}
  {$ENDIF}
{$ELSEIF Defined(UNIX)}
{$IFEND}
begin
  Self.IOselector_ChangeHostAPI( ComboBox1, ComboBox2, ComboBox3, Pa_GetDefaultHostApi());
end;


procedure TBasicPortAudioIO.IOselector_ChangeHostAPI( ComboBox1,
                                                      ComboBox2,
                                                      ComboBox3:TComboBox;
                                                      newHostAPIindex:TPaHostApiIndex);
var
  DeviceName  : String;
  DeviceCount : Integer;
  DeviceIndex : cint;    // = TPaDeviceIndex;
  i           : Integer;
begin
  // DRAGONS - probably test on assigned( ComboBox1) here .. ???

  ComboBox1.Clear;  // remove all existing items, we rebuild the list now !
  for i:= 0 to (Self.paConfig.HOST_API_COUNT -1)
    do ComboBox1.AddItem( Self.Str_PA_HostAPIName( i), NIL);

  // as ComboBox1 is cleared, we have to set again the Selected API here !
  ComboBox1.ItemIndex:= newHostAPIindex;

  ComboBox2.Clear;  // remove all existing items, we rebuild the list now !
  ComboBox3.Clear;  // remove all existing items, we rebuild the list now !

  i:= 0;
  DeviceCount:= Self.PA_Host_DeviceCount( newHostAPIindex);
  if DeviceCount > 0
    then
      begin
        // ok, so update this value :
        Self.PA_HostAPI:= newHostAPIindex;  // added 2021-03-08

        // very important bugfix 2021-02-28 :
        paConfig.PA_INPUT_DEVICE:= paNoDevice;

        for i:= 0 to DeviceCount -1 do
          begin
            DeviceIndex:= Self.PA_Device_Index( newHostAPIindex, i);
            DeviceName:= Self.Str_PA_DeviceName( DeviceIndex);
            //ShowMessage( 'DEBUG' +#13 +DeviceName);

            { only add inputs to this ComboBox : }
            if Self.PA_Device_NumInputChannels( DeviceIndex) > 0
              then ComboBox2.AddItem( IntToStr( DeviceIndex) +' = ' +DeviceName, NIL);

            { only add outputs to this ComboBox : }
            if Self.PA_Device_NumOutputChannels( DeviceIndex) > 0
              then ComboBox3.AddItem( IntToStr( DeviceIndex) +' = ' +DeviceName, NIL);
          end;

        { reset the item index to initial position : }
        Self.IOselector_ChangeInputDevice( ComboBox2);
        Self.IOselector_ChangeOutputDevice( ComboBox3);

        // update the Stream Data !
        Self.Update_Host_API( newHostAPIindex);  // added 2021-03-21
      end;
end;


function iReadValueFromStringStart( s:string):integer;
var
  i : integer;
  z : String;
begin
  // This function even works with an "empty string" = '',
  // In this case, the Result is := 0 !
  Result:= 0;

  for i:= 1 to Length( s) do
    case s[i] of
      '0'..'9': // found a number
        begin
          z:= s[i];
          Result:= Result * 10 +StrToInt( z);
        end;
      else Break;
    end;
  //ShowMessage( 'ReadValueFromStringStart = ' +IntToStr( Result));
end;


procedure TBasicPortAudioIO.IOselector_ChangeInputDevice( var ComboBoxInput:TComboBox);
begin
  // extract NumberOfDevice from Text
  if ComboBoxInput.Items.Count > 0 then
    begin
      if ComboBoxInput.ItemIndex < 0 then ComboBoxInput.ItemIndex:= 0;
      Self.PA_InputDevice:= iReadValueFromStringStart( ComboBoxInput.Items[ComboBoxInput.ItemIndex]);
    end;
end;


procedure TBasicPortAudioIO.IOselector_ChangeOutputDevice( var ComboBoxOutput:TComboBox);
begin
  // extract NumberOfDevice from Text
  if ComboBoxOutput.Items.Count > 0 then
    begin
      if ComboBoxOutput.ItemIndex < 0 then ComboBoxOutput.ItemIndex:= 0;
      Self.PA_OutputDevice:= iReadValueFromStringStart( ComboBoxOutput.Items[ComboBoxOutput.ItemIndex]);
    end;
end;


end.

