unit DemoPortaudioIOobject;

{$include RELEASE_SETTINGS.INC}

{$IFDEF FPC}
  {$mode objfpc}{$H+}
{$ENDIF}

{$IFDEF DEBUG_VERSION}
  //{$DEFINE DEBUG_THIS_UNIT}
{$ENDIF}

{*******************************************************************************
*                    DemoPortaudioIOobject.pas                                 *
*                                                                              *
*      Demo Object showing interleaved stereo audio I/O                        *
*                                                                              *
*                    Created 2021-02-16 by BREAKOUTBOX                         *
*                    Latest changes:  2024-12-03                               *
*******************************************************************************}

interface

uses
  Interfaces,
  LCLIntf,  // for PostMessage() ..
  Classes, Controls, Forms, Dialogs, //StdCtrls,
  ExtCtrls, SysUtils, CTypes,
  portaudio, TypesAndConst,
  BasicPortAudioIOobject;


{ ---------------------------------------------------------------------------- }
{            Define some application specific error codes here                 }
{                                                                              }
{ Please take into account that Portaudio uses ( -10000 +x )  for errors       }
{ Also check error codes of LibSndFile, LibSamplerate etc ..                   }
{ ---------------------------------------------------------------------------- }
const
  DemoErr_Base  = -1000;
  DemoErr_01    =  DemoErr_Base  +1;
  DemoErr_02    =  DemoErr_Base  +2;
  DemoErr_03    =  DemoErr_Base  +3;
  DemoErr_04    =  DemoErr_Base  +4;
  DemoErr_05    =  DemoErr_Base  +5;
  // ...
  DemoErr_99    =  DemoErr_Base +99;


type
  TDemoPortAudioIO = class(TBasicPortAudioIO)
  private
    {private fields}
    //
    {private methods}
  protected
    {protected fields}
    TrackBarVolume : integer;
    {protected methods}
    procedure CallBack_ProcessBuffers( pInputBuffer, pOutputBuffer:pointer); reintroduce;
  public
    {public fields}
    //
    {public methods}
    Constructor Create(AOwner: TComponent; CallbackAdress:Pointer); reintroduce;
    Destructor  Destroy; override;
  published
    { published declarations }
    property Volume:integer write TrackBarVolume;
  end;



{ This routine will be called by the PortAudio engine when audio is needed.
  It may be called at interrupt level on some machines so don't do anything
  that could mess up the system like calling malloc() or free().             }
function PortAudioCallback( const pInputBuffer         : pointer;
                            pOutputBuffer              : pointer;
	                    const {%H-}FramesPerBuffer : culong;
	                    const {%H-}pTimeInfo       : PPaStreamCallbackTimeInfo;
	                    {%H-}PAStatusFlags         : TPaStreamCallbackFlags;
	                    pUserData                  : pointer) : CInt32; cdecl;


implementation

uses
  LConvEncoding,
  LazUTF8;


var
  iPortAudioCallback_DoNotEnter : boolean;

// This routine will be called by the PortAudio engine when audio is needed.
//   It may be called at interrupt level on some machines so don't do anything
//   that could mess up the system like calling malloc() or free().
function PortAudioCallback( const pInputBuffer    : pointer;
                            pOutputBuffer         : pointer;
	                    const FramesPerBuffer : culong;
	                    const pTimeInfo       : PPaStreamCallbackTimeInfo;
	                    PAStatusFlags         : TPaStreamCallbackFlags;
	                    pUserData             : pointer) : CInt32; cdecl;
var
  pObject : TDemoPortAudioIO;
begin
  // The stream callback should return one of the values in the
  // PaStreamCallbackResult enumeration. To ensure that the callback
  // continues to be called, it should return paContinue (0)
  Result:= paContinue;


  // pUserData is a pointer to any data You need in the callback
  // In this example I give the callback access to my TDemoPortAudioIO object
  pObject:= TDemoPortAudioIO( pUserData);


  // save actual values "CpuLoad" and "CurrentTime" from PortAudio
  pObject.paConfig.CpuLoad:= PA_GetStreamCpuLoad( pObject.paConfig.pStream);
  pObject.paConfig.PA_Time:= Round( pTimeInfo^.CurrentTime);

  // inform the main form about PA errors :
  if PAStatusFlags <> 0      // send PA error status flags to MainForm !
    then PostMessage( Application.Mainform.Handle,
                      LM_PA_STATUSFLAGS, PAStatusFlags, pObject.paConfig.PA_Time);


  //----------------------------------------------------------------------------
  // Protect against "Overload" - avoid accessing the code more than once :
  //   ( indeed, this should NEVER happen at all, but who knows .. )
  if iPortAudioCallback_DoNotEnter = TRUE then Exit;

  // on "FALSE" do DSP Your processing now ..
  iPortAudioCallback_DoNotEnter:= TRUE;

  // exit here if we don't run STEREO :    ( this is my design decision ! )
  if (pObject.paConfig.InputParameters.channelCount <> 2)
    or (pObject.paConfig.OutputParameters.channelCount <> 2) then
    begin
      Result:= -1;   // DRAGONS - error undefined
      Exit;
    end;


  // ---------------------------------------------------------------------------
  // HERE we "link in" our AudioDSP processing
  if Assigned( pObject)
    then pObject.CallBack_ProcessBuffers( pInputBuffer, pOutputBuffer);
  // ---------------------------------------------------------------------------


  // copy actual buffer pointer to a global pointer, for VU Meter etc. :
  pGOutputBuffer:= @pOutputBuffer^;
  // DRAGONS:  better COPY the content to a global tempBuffer, pointer could get invalid .. !

  iPortAudioCallback_DoNotEnter:= FALSE;
  //----------------------------------------------------------------------------
end;


procedure TDemoPortAudioIO.CallBack_ProcessBuffers( pInputBuffer,
                                                    pOutputBuffer:pointer);
var
  pINPUT  : PINPUT_SAMPLE;
  pOUTPUT : POUTPUT_SAMPLE;
  i, n    : Dword;
begin
  // --- set Input Pointer ---
  // PA CallBack Function may get called with NULL inputBuffer,
  // during initial setup, or because there IS no input device !
  if pInputBuffer = NIL
     then pINPUT:= @paConfig.SilentBuf[0]    // read a dummy buffer
     else pINPUT:= pInputBuffer;

  // --- set Output Pointer ---
  pOUTPUT:= pOutputBuffer;


  // ---------------------------------------------------------------------------
  // insert Your code here to insert Audio from File or do DSP ..
  // - pINPUT  is your source
  // - pOUTPUT is your target
  //
  // for example ..
  n:= paConfig.FramesPerCallback * 2;
  for i:= 1 to n do
    begin
      // simply copy both channels (interleaved data) from IN to OUT
      {$ifdef PORTAUDIO_USE_FLOAT_IO}   // if activated, we use float32 instead of int16 !
      pOUTPUT^:= pINPUT^ * TrackBarVolume / 255;
      {$ELSE}
      pOUTPUT^:= Round( pINPUT^ * TrackBarVolume / 255);
      {$ENDIF}
      inc( pINPUT);
      inc( pOUTPUT);
    end;
  // ---------------------------------------------------------------------------
end;


// -----------------------------------------------------------------------------
// ----------------- Create ----------------------------------------------------
// -----------------------------------------------------------------------------
constructor TDemoPortAudioIO.Create(AOwner: TComponent; CallbackAdress:Pointer);

begin
  // init Your settings, for example ..
  TrackBarVolume:= 255;

  inherited Create( AOwner, CallbackAdress, PA_STREAMTYPE_STEREO);
end;


// ---------------- Destroy ----------------------------------------------------
Destructor TDemoPortAudioIO.Destroy;
begin
  // do something on destroy

  inherited Destroy;
end;


end.

