unit PortaudioAPIselector;

{$include RELEASE_SETTINGS.INC}

{$IFDEF FPC}
  {$mode objfpc}{$H+}
{$ENDIF}

{$IFDEF DEBUG_VERSION}
  {$DEFINE DEBUG_THIS_UNIT}    //activate this to get debug messages from this unit !
  // You can comment it out if You don't want Debug messages from this Unit !
{$ENDIF}

{*******************************************************************************
*                     PortaudioAPIselector.pas
*
*                     created  2012-01-20
*                     latest changes  2025-12-05
********************************************************************************
*  Informations about WASAPI:  https://markheath.net/post/what-up-with-wasapi
*  Informations about WDM (Windows Driver Model Kernel Streaming):
*******************************************************************************}

interface

uses
  Interfaces,     // this includes the LCL widgetset
  LConvEncoding,  // for ISO_8859_15ToUTF8()
  JWAwindows,     // for PostMessage

  Forms,
  Classes,
  Controls,
  StdCtrls,   // for TMemo
  ExtCtrls,
  Dialogs,
  SysUtils,
  Graphics,   // for clInactiveBorder
  Basicportaudioioobject,  // this unit MUST BE before DemoPortaudioIOobject !!!
                           // otherwise malfunction !!
  {$IFDEF USE_TDemoPortAudioIO}
  //DemoPortaudioIOobject,
  {$ENDIF}
  DebugIntoMemo,
  ctypes;


var  // moved here 2021-05-08 ..
  // a global VAR pointer to the DebugMemo :
  pToLogfileMemo : TMemo = NIL;


// ---------------------------------------------------------------
// Select the PORTAUDIO Host API and Devices with this Component :
// ---------------------------------------------------------------

type
  TPortaudioAPIselector = class(TPanel)
  private
    { private declarations }
    MyParentsHandle : THandle;  // added 2021-03-19
    KeepMeVisible   : boolean;  // on FALSE, the APIselector behaves like a popup
  protected
    { protected declarations }
    pToPAObject : TBasicPortAudioIO;  // added 2021-03-18
    ButtonEditAudioSettings : TButton;
    ComboBoxHostAPI : TComboBox;
    ComboBoxInput   : TComboBox;
    ComboBoxOutput  : TComboBox;
    LabelHost   : TLabel;
    LabelInput  : TLabel;
    LabelOutput : TLabel;
    procedure ButtonEditAudioSettingsClick(Sender: TObject);
    procedure ComboBoxHostAPIChange(Sender:TObject);
    procedure ComboBoxInputChange(Sender:TObject);
    procedure ComboBoxOutputChange(Sender:TObject);
    procedure Enable( b:boolean);
    { --- update ComboBoxes via OnChange Event Handler --- }
    procedure HostAPIChange;
    procedure InputDeviceChange;
    procedure OutputDeviceChange;
  public
    constructor Create( TheOwner: TComponent;
                        TheParent:TWinControl;
                        OriginX, OriginY:integer;
                        KeepVisible:boolean;
                        PAObject : TBasicPortAudioIO); reintroduce;
    Destructor  Destroy; Override;
  end;



implementation


uses
  portaudio,
  TypesAndConst;


{ ----- TPortaudioAPIselector ------------------------------------------------ }
constructor TPortaudioAPIselector.Create( TheOwner: TComponent;
                                          TheParent:TWinControl;
                                          OriginX, OriginY:integer;
                                          KeepVisible:boolean;
                                          PAObject : TBasicPortAudioIO);
begin
  inherited Create( TheOwner);

  // keep the handle of the parent ( "Self.Parent.Handle" didin't work ..)
  MyParentsHandle:= TheParent.Handle;

  // get permanent access to Portaudio Object :
  pToPAObject:= PAObject;  // added 2021-03-18

  KeepMeVisible := KeepVisible;
  Self.Visible  := KeepVisible;

  Self.Parent := TheParent;
  Self.Left   := Scale96ToForm( OriginX);
  Self.Top    := Scale96ToForm( OriginY);
  Self.Width  := Scale96ToForm( 250);
  Self.Height := Scale96ToForm( 170);
  Self.Color  := clSilver;
  // Remark:  Scale96ToForm() crashes the app if there's no valid Parent !

  ButtonEditAudioSettings:= TButton.Create( Self);
  with ButtonEditAudioSettings do
    begin
      Parent  := Self;
      Left    := Scale96ToForm( 8);
      Top     := Scale96ToForm( 8);
      Width   := Self.Width -Scale96ToForm( 16);
      Height  := Scale96ToForm( 26);
      Caption := 'Edit Audio Settings';
      OnClick:= @ButtonEditAudioSettingsClick;
    end;

  // -----------------------------------------
  LabelHost   := TLabel.Create( Self);
  LabelInput  := TLabel.Create( Self);
  LabelOutput := TLabel.Create( Self);

  LabelHost.Parent   := Self;
  LabelInput.Parent  := Self;
  LabelOutput.Parent := Self;

  LabelHost.caption   := 'Audio Interface (API)';
  LabelInput.caption  := 'Input';
  LabelOutput.caption := 'Output';

  LabelHost.Top   := Scale96ToForm( 38);
  LabelInput.Top  := Scale96ToForm( 38 +42);
  LabelOutput.Top := Scale96ToForm( 38 +42 * 2);

  LabelHost.Left   := Scale96ToForm( 8);
  LabelInput.Left  := Scale96ToForm( 8);
  LabelOutput.Left := Scale96ToForm( 8);

  LabelHost.Font.Size   := 9;
  LabelInput.Font.Size  := 9;
  LabelOutput.Font.Size := 9;

  // -----------------------------------------
  ComboBoxHostAPI := TComboBox.Create( Self);
  ComboBoxInput   := TComboBox.Create( Self);
  ComboBoxOutput  := TComboBox.Create( Self);

  ComboBoxHostAPI.Parent := Self;
  ComboBoxInput.Parent   := Self;
  ComboBoxOutput.Parent  := Self;

  ComboBoxHostAPI.Left := Scale96ToForm( 8);
  ComboBoxInput.Left   := Scale96ToForm( 8);
  ComboBoxOutput.Left  := Scale96ToForm( 8);

  ComboBoxHostAPI.Top := Scale96ToForm( 54);
  ComboBoxInput.Top   := Scale96ToForm( 54 +42);
  ComboBoxOutput.Top  := Scale96ToForm( 54 +42 * 2);

  ComboBoxHostAPI.Width := Self.Width -Scale96ToForm( 16);
  ComboBoxInput.Width   := Self.Width -Scale96ToForm( 16);
  ComboBoxOutput.Width  := Self.Width -Scale96ToForm( 16);

  ComboBoxHostAPI.Style := csDropDownList;
  ComboBoxInput.Style   := csDropDownList;
  ComboBoxOutput.Style  := csDropDownList;

  ComboBoxHostAPI.OnChange := @ComboBoxHostAPIChange;
  ComboBoxInput.OnChange   := @ComboBoxInputChange;
  ComboBoxOutput.OnChange  := @ComboBoxOutputChange;

  Self.Enable( FALSE);

  if assigned( PAObject)
    then PAObject.IOselector_init( ComboBoxHostAPI, ComboBoxInput, ComboBoxOutput);
end;


destructor TPortaudioAPIselector.Destroy;
begin
  // You've passed Your form as owner of these Objects, don't free here !
  // --------------------------------------------------------------------
  //LabelHost.Free;
  //LabelInput.Free;
  //LabelOutput.Free;
  //ComboBoxHostAPI.Free;
  //ComboBoxInput.Free;
  //ComboBoxOutput.Free;

  inherited Destroy;
end;


procedure TPortaudioAPIselector.Enable( b:boolean);
begin
  ComboBoxHostAPI.Enabled := b;
  ComboBoxInput.Enabled   := b;
  ComboBoxOutput.Enabled  := b;

  LabelHost.Font.Bold   := b;
  LabelInput.Font.Bold  := b;
  LabelOutput.Font.Bold := b;

  // Parent has to react on this message, probably ..
  if b
    then PostMessage( MyParentsHandle, LM_PA_APISELECTOR, WPARAM( TRUE), 0)
    else PostMessage( MyParentsHandle, LM_PA_APISELECTOR, WPARAM( FALSE), 0);
end;


{ --- Button Actions --------------------------------------------------------- }
procedure TPortaudioAPIselector.ButtonEditAudioSettingsClick( Sender: TObject);
var
  err : cint; // = TPaError
begin
  if assigned( pToPAObject) then
    begin
      if pToPAObject.PA_IsRunning
        then
          begin
            err:= pToPAObject.PA_stop_and_close_Stream;
            Self.ButtonEditAudioSettings.Caption:= 'Activate new Settings';
            Self.Color:= clInactiveBorder;
            Self.Enable( TRUE);
            // post a Message here, to MainForm: "LM_PA_STREAM_CLOSED"
            PostMessage( MyParentsHandle, LM_PA_APISELECTOR,
                                          LM_PA_APISELECTOR_CHANGE,
                                          LM_PA_STREAM_CLOSED);
          end

        else
          begin
            err:= pToPAObject.PA_open_and_start_Stream;
            if err = NO_ERROR
              then
                begin
                  Self.ButtonEditAudioSettings.Caption:= 'Edit Audio Settings';
                  Self.Color:= clSilver; //clDefault;
                  Self.Enable( FALSE);
                  if not KeepMeVisible then Self.Visible:= FALSE;

                  // post a Message here, to MainForm: "LM_PA_STREAM_OPENED"
                  PostMessage( MyParentsHandle, LM_PA_APISELECTOR,
                                                LM_PA_APISELECTOR_CHANGE,
                                                LM_PA_STREAM_OPENED);
                  // Im WPARAM übergibt man üblichwerweise Integer-Werte oder Handles,
                  // während hingegegen im LPARAM üblicherweise Zeiger übergeben werden.
                  // http://michael-puff.de/Programmierung/Artikel/WParam_LParam.shtml
                end

              else
                begin
                  // error - for example - happens when a WDM-KS device is
                  // already in use exclusively by another application
                  ShowMessage( 'PortaudioAPIselector.pas:  Error while PA_open_and_start_Stream() = '
                               +IntToStr( err) +LineEnding +LineEnding
                               +pToPAObject.Str_PA_HostAPI_GetLastHostErrorText);
                end;
          end;
    end;
end;


{ --- ComboBox Actions ------------------------------------------------------- }
procedure TPortaudioAPIselector.ComboBoxHostAPIChange(Sender:TObject);
begin
  Self.HostAPIChange;
end;


procedure TPortaudioAPIselector.ComboBoxInputChange(Sender:TObject);
begin
  InputDeviceChange;
end;


procedure TPortaudioAPIselector.ComboBoxOutputChange(Sender:TObject);
begin
  OutputDeviceChange;
end;


{ ---------------------------------------------------------------------------- }
{ --------------------- CONFIGURE PortAudio API ------------------------------ }
{ ---------------------------------------------------------------------------- }

{ *** update ComboBoxes via OnChange Event Handler : ************************* }
procedure TPortaudioAPIselector.HostAPIChange;
begin
  // try with ComboBoxHostAPI Item Index :
  if pToPAObject.PA_Host_DeviceCount( ComboBoxHostAPI.ItemIndex) > 0
    then
      begin
        // we have to update ALL three ComboBoxes :
        pToPAObject.IOselector_ChangeHostAPI( ComboBoxHostAPI, ComboBoxInput,
                                              ComboBoxOutput, ComboBoxHostAPI.ItemIndex);

        Memo_ShowPAVersionInformations( TBasicPortAudioIO( pToPAObject),
                                        pToLogfileMemo,
                                        portaudio.LibName);
      end

    else
      begin
        ShowMessage( 'Audio Interface  "'
                     +pToPAObject.Str_PA_HostAPIName( ComboBoxHostAPI.ItemIndex)
                     +'"  exists - but has no devices, sorry.');
        // failed, reset old status :
        ComboBoxHostAPI.ItemIndex:= pToPAObject.PA_HostAPI;
      end;
end;


procedure TPortaudioAPIselector.InputDeviceChange;
begin
  pToPAObject.IOselector_ChangeInputDevice( ComboBoxInput);
  Memo_ShowPAVersionInformations( TBasicPortAudioIO( pToPAObject),
                                  pToLogfileMemo,
                                  portaudio.LibName);
end;


procedure TPortaudioAPIselector.OutputDeviceChange;
begin
  pToPAObject.IOselector_ChangeOutputDevice( ComboBoxOutput);
  Memo_ShowPAVersionInformations( TBasicPortAudioIO( pToPAObject),
                                  pToLogfileMemo,
                                  portaudio.LibName);
end;


end.
