unit portaudio;

// With this switch activated You can alternatively
// load the Portaudio.dll on runtime ("late binding") :

//{$define LOAD_PA_ON_RUNTIME}


{*
* $Id$
* PortAudio Portable Real-Time Audio Library
* PortAudio API Header File
* Latest version available at: http://www.portaudio.com/
*
* Copyright (c) 1999-2002 Ross Bencina and Phil Burk
*
* Permission is hereby granted, free of charge, to any person obtaining
* a copy of this software and associated documentation files
* (the "Software"), to deal in the Software without restriction,
* including without limitation the rights to use, copy, modify, merge,
* publish, distribute, sublicense, and/or sell copies of the Software,
* and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
*
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
* ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
* CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
* WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *}

{*
* The text above constitutes the entire PortAudio license; however,
* the PortAudio community also makes the following non-binding requests:
*
* Any person wishing to distribute modifications to the Software is
* requested to send the modifications to the original developer so that
* they can be incorporated into the canonical version. It is also
* requested that these non-binding requests be included along with the
* license above.
 *}

{*******************************************************************************
*                           File History
*
* - 2015-04-13  added support for loading the DLL on runtime
*               based on the work of Fred van Stappen  => uos_portaudio.pas
* - 2024-07-07  added some values from current portaudio.h
* - 2025-12-04  made some changes to support both 32 and 64 bit DLLs
*******************************************************************************}


{$IFDEF FPC}
  {$PACKENUM 4}    (* use 4-byte enums *)
  {$PACKRECORDS C} (* C/C++-compatible record packing *)
{$ELSE}  // = Delphi
  {$MINENUMSIZE 4} (* use 4-byte enums *)
{$ENDIF}

interface

uses
  {$ifdef LOAD_PA_ON_RUNTIME}
  dynlibs,
  {$endif}
  cTypes;

const
{$IF Defined(MSWINDOWS)}
  {$IF Defined(WIN32)}
  LibName = 'portaudio32.dll';
  {$ELSE}
  LibName = 'portaudio64.dll';
  {$ENDIF}
{$ELSEIF Defined(DARWIN)}
  // this is for portaudio version 19
  LibName = 'libportaudio.2.dylib';
  {$LINKLIB libportaudio.2}
{$ELSEIF Defined(UNIX)}
  LibName = 'libportaudio.so';
{$IFEND}



{** Error codes returned by PortAudio functions.
 Note that with the exception of paNoError, all PaErrorCodes are negative.
*}

type TPaError = cInt;
type TPaErrorCode = cInt; const
    paNoError = 0;

    paNotInitialized = -10000;
    paUnanticipatedHostError                = (paNotInitialized+ 1);
    paInvalidChannelCount                   = (paNotInitialized+ 2);
    paInvalidSampleRate                     = (paNotInitialized+ 3);
    paInvalidDevice                         = (paNotInitialized+ 4);
    paInvalidFlag                           = (paNotInitialized+ 5);
    paSampleFormatNotSupported              = (paNotInitialized+ 6);
    paBadIODeviceCombination                = (paNotInitialized+ 7);
    paInsufficientMemory                    = (paNotInitialized+ 8);
    paBufferTooBig                          = (paNotInitialized+ 9);
    paBufferTooSmall                        = (paNotInitialized+10);
    paNullCallback                          = (paNotInitialized+11);
    paBadStreamPtr                          = (paNotInitialized+12);
    paTimedOut                              = (paNotInitialized+13);
    paInternalError                         = (paNotInitialized+14);
    paDeviceUnavailable                     = (paNotInitialized+15);
    paIncompatibleHostApiSpecificStreamInfo = (paNotInitialized+16);
    paStreamIsStopped                       = (paNotInitialized+17);
    paStreamIsNotStopped                    = (paNotInitialized+18);
    paInputOverflowed                       = (paNotInitialized+19);
    paOutputUnderflowed                     = (paNotInitialized+20);
    paHostApiNotFound                       = (paNotInitialized+21); // The notes below are from the
    paInvalidHostApi                        = (paNotInitialized+22); // original file portaudio.h
    paCanNotReadFromACallbackStream         = (paNotInitialized+23); {**< @todo review error code name *}
    paCanNotWriteToACallbackStream          = (paNotInitialized+24); {**< @todo review error code name *}
    paCanNotReadFromAnOutputOnlyStream      = (paNotInitialized+25); {**< @todo review error code name *}
    paCanNotWriteToAnInputOnlyStream        = (paNotInitialized+26); {**< @todo review error code name *}
    paIncompatibleStreamHostApi             = (paNotInitialized+27);
    paBadBufferPtr                          = (paNotInitialized+28);
    paCanNotInitializeRecursively           = (paNotInitialized+29); // added 2024-07-07


{** The type used to enumerate to host APIs at runtime. Values of this type
 range from 0 to (Pa_GetHostApiCount()-1).

 @see Pa_GetHostApiCount
*}
type TPaHostApiIndex = cInt;


  {** Unchanging unique identifiers for each supported host API. This type
   is used in the PaHostApiInfo structure. The values are guaranteed to be
   unique and to never change, thus allowing code to be written that
   conditionally uses host API specific extensions.

   New type ids will be allocated when support for a host API reaches
   "public alpha" status, prior to that developers should use the
   paInDevelopment type id.

   @see PaHostApiInfo
  *}
  type TPaHostApiTypeId = cInt; const
      paInDevelopment   = 0; {* use while developing support for a new host API *}
      paDirectSound     = 1;
      paMME             = 2;
      paASIO            = 3;
      paSoundManager    = 4;
      paCoreAudio       = 5;
      paOSS             = 7;
      paALSA            = 8;
      paAL              = 9;
      paBeOS            = 10;
      paWDMKS           = 11;
      paJACK            = 12;
      paWASAPI          = 13;
      paAudioScienceHPI = 14;
      paAudioIO         = 15;  // 2024-07-07 added from current portaudio.h
      paPulseAudio      = 16;  // 2024-07-07 added from current portaudio.h
      paSndio           = 17;  // 2024-07-07 added from current portaudio.h


  {** The type used to refer to audio devices. Values of this type usually
   range from 0 to (Pa_GetDeviceCount()-1), and may also take on the PaNoDevice
   and paUseHostApiSpecificDeviceSpecification values.

   @see Pa_GetDeviceCount, paNoDevice, paUseHostApiSpecificDeviceSpecification
  *}
  type TPaDeviceIndex = cInt;


{** A structure containing information about a particular host API. *}
type
  PPaHostApiInfo = ^TPaHostApiInfo;
  TPaHostApiInfo = record
      {** this is struct version 1 *}
      structVersion: cInt;
      {** The well known unique identifier of this host API @see PaHostApiTypeId *}
      _type: TPaHostApiTypeId;
      {** A textual description of the host API for display on user interfaces. *}
      name: pAnsiChar;

      {**  The number of devices belonging to this host API. This field may be
       used in conjunction with Pa_HostApiDeviceIndexToDeviceIndex() to enumerate
       all devices for this host API.
       @see Pa_HostApiDeviceIndexToDeviceIndex
      *}
      deviceCount: cInt;

      {** The default input device for this host API. The value will be a
       device index ranging from 0 to (Pa_GetDeviceCount()-1), or paNoDevice
       if no default input device is available.
      *}
      defaultInputDevice: TPaDeviceIndex;

      {** The default output device for this host API. The value will be a
       device index ranging from 0 to (Pa_GetDeviceCount()-1), or paNoDevice
       if no default output device is available.
      *}
      defaultOutputDevice: TPaDeviceIndex;
  end;


  {** Structure used to return information about a host error condition.
  *}
  type
    PPaHostErrorInfo = ^TPaHostErrorInfo;
    TPaHostErrorInfo = record
        hostApiType: TPaHostApiTypeId;    {**< the host API which returned the error code *}
        errorCode: clong;                 {**< the error code returned *}
        errorText: pAnsiChar;                 {**< a textual description of the error if available, otherwise a zero-length string *}
    end;


{** The type used to represent monotonic time in seconds that can be used
    for syncronisation. The type is used for the outTime argument to the
    PaStreamCallback and as the result of Pa_GetStreamTime().

    @see PaStreamCallback, Pa_GetStreamTime
*}
type TPaTime = cdouble;


{** A structure providing information and capabilities of PortAudio devices.
    Devices may support input, output or both input and output.
*}
type
  PPaDeviceInfo = ^TPaDeviceInfo;
  TPaDeviceInfo = record
      structVersion: cInt;  {* this is struct version 2 *}
      name: pAnsiChar;
      hostApi: TPaHostApiIndex; {* note this is a host API index, not a type id*}

      maxInputChannels: cInt;
      maxOutputChannels: cInt;

      {* Default latency values for interactive performance. *}
      defaultLowInputLatency: TPaTime;
      defaultLowOutputLatency: TPaTime;
      {* Default latency values for robust non-interactive applications (eg. playing sound files). *}
      defaultHighInputLatency: TPaTime;
      defaultHighOutputLatency: TPaTime;

      defaultSampleRate: cdouble;
 end;


{** A type used to specify one or more sample formats. Each value indicates
 a possible format for sound data passed to and from the stream callback,
 Pa_ReadStream and Pa_WriteStream.

 The standard formats paFloat32, paInt16, paInt32, paInt24, paInt8
 and aUInt8 are usually implemented by all implementations.

 The floating point representation (paFloat32) uses +1.0 and -1.0 as the
 maximum and minimum respectively.

 paUInt8 is an unsigned 8 bit format where 128 is considered "ground"

 The paNonInterleaved flag indicates that a multichannel buffer is passed
 as a set of non-interleaved pointers.

 @see Pa_OpenStream, Pa_OpenDefaultStream, PaDeviceInfo
 @see paFloat32, paInt16, paInt32, paInt24, paInt8
 @see paUInt8, paCustomFormat, paNonInterleaved
*}
type TPaSampleFormat = culong;
const
  paFloat32        = TPaSampleFormat($00000001); {**< @see PaSampleFormat *}
  paInt32          = TPaSampleFormat($00000002); {**< @see PaSampleFormat *}
  paInt24          = TPaSampleFormat($00000004); {**< Packed 24 bit format. @see PaSampleFormat *}
  paInt16          = TPaSampleFormat($00000008); {**< @see PaSampleFormat *}
  paInt8           = TPaSampleFormat($00000010); {**< @see PaSampleFormat *}
  paUInt8          = TPaSampleFormat($00000020); {**< @see PaSampleFormat *}
  paCustomFormat   = TPaSampleFormat($00010000); {**< @see PaSampleFormat *}
  paNonInterleaved = TPaSampleFormat($80000000);


{** Parameters for one direction (input or output) of a stream.
*}
type
  PPaStreamParameters = ^TPaStreamParameters;
  TPaStreamParameters = record
      {** A valid device index in the range 0 to (Pa_GetDeviceCount()-1)
       specifying the device to be used or the special constant
       paUseHostApiSpecificDeviceSpecification which indicates that the actual
       device(s) to use are specified in hostApiSpecificStreamInfo.
       This field must not be set to paNoDevice.
      *}
      device: TPaDeviceIndex;

      {** The number of channels of sound to be delivered to the
       stream callback or accessed by Pa_ReadStream() or Pa_WriteStream().
       It can range from 1 to the value of maxInputChannels in the
       PaDeviceInfo record for the device specified by the device parameter.
      *}
      channelCount: cInt;

      {** The sample format of the buffer provided to the stream callback,
       a_ReadStream() or Pa_WriteStream(). It may be any of the formats described
       by the PaSampleFormat enumeration.
      *}
      sampleFormat: TPaSampleFormat;

      {** The desired latency in seconds. Where practical, implementations should
       configure their latency based on these parameters, otherwise they may
       choose the closest viable latency instead. Unless the suggested latency
       is greater than the absolute upper limit for the device implementations
       should round the suggestedLatency up to the next practial value - ie to
       provide an equal or higher latency than suggestedLatency wherever possibe.
       Actual latency values for an open stream may be retrieved using the
       inputLatency and outputLatency fields of the PaStreamInfo structure
       returned by Pa_GetStreamInfo().
       @see default*Latency in PaDeviceInfo, *Latency in PaStreamInfo
      *}
      suggestedLatency: TPaTime;

      {** An optional pointer to a host api specific data structure
       containing additional information for device setup and/or stream processing.
       hostApiSpecificStreamInfo is never required for correct operation,
       if not used it should be set to NULL.
      *}
      hostApiSpecificStreamInfo: Pointer;
  end;


{**
 A single PaStream can provide multiple channels of real-time
 streaming audio input and output to a client application. A stream
 provides access to audio hardware represented by one or more
 PaDevices. Depending on the underlying Host API, it may be possible
 to open multiple streams using the same device, however this behavior
 is implementation defined. Portable applications should assume that
 a PaDevice may be simultaneously used by at most one PaStream.

 Pointers to PaStream objects are passed between PortAudio functions that
 operate on streams.

 @see Pa_OpenStream, Pa_OpenDefaultStream, Pa_OpenDefaultStream, Pa_CloseStream,
 Pa_StartStream, Pa_StopStream, Pa_AbortStream, Pa_IsStreamActive,
 Pa_GetStreamTime, Pa_GetStreamCpuLoad

*}
type
  PPaStream = Pointer;


{** Can be passed as the framesPerBuffer parameter to Pa_OpenStream()
 or Pa_OpenDefaultStream() to indicate that the stream callback will
 accept buffers of any size.
*}
const paFramesPerBufferUnspecified = (0);


{** Flags used to control the behavior of a stream. They are passed as
 parameters to Pa_OpenStream or Pa_OpenDefaultStream. Multiple flags may be
 ORed together.

 @see Pa_OpenStream, Pa_OpenDefaultStream
 @see paNoFlag, paClipOff, paDitherOff, paNeverDropInput,
  paPrimeOutputBuffersUsingStreamCallback, paPlatformSpecificFlags
*}
type TPaStreamFlags = culong;

{** @see PaStreamFlags *}
const   paNoFlag          = TPaStreamFlags(0);

{** Disable default clipping of out of range samples.
 @see PaStreamFlags
*}
const   paClipOff         = TPaStreamFlags($00000001);

{** Disable default dithering.
 @see PaStreamFlags
*}
const   paDitherOff       = TPaStreamFlags($00000002);

{** Flag requests that where possible a full duplex stream will not discard
 overflowed input samples without calling the stream callback. This flag is
 only valid for full duplex callback streams and only when used in combination
 with the paFramesPerBufferUnspecified (0) framesPerBuffer parameter. Using
 this flag incorrectly results in a paInvalidFlag error being returned from
 Pa_OpenStream and Pa_OpenDefaultStream.

 @see PaStreamFlags, paFramesPerBufferUnspecified
*}
const   paNeverDropInput  = TPaStreamFlags($00000004);

{** Call the stream callback to fill initial output buffers, rather than the
 default behavior of priming the buffers with zeros (silence). This flag has
 no effect for input-only and blocking read/write streams.

 @see PaStreamFlags
*}
const   paPrimeOutputBuffersUsingStreamCallback = TPaStreamFlags($00000008);

{** A mask specifying the platform specific bits.
 @see PaStreamFlags
*}
const   paPlatformSpecificFlags = TPaStreamFlags($FFFF0000);

{**
 Timing information for the buffers passed to the stream callback.
*}


type
  PPaStreamCallbackTimeInfo = ^TPaStreamCallbackTimeInfo;
  TPaStreamCallbackTimeInfo = record
      inputBufferAdcTime: TPaTime;
      currentTime: TPaTime;
      outputBufferDacTime: TPaTime;
  end;


{**
 Flag bit constants for the statusFlags to PaStreamCallback.

 @see paInputUnderflow, paInputOverflow, paOutputUnderflow, paOutputOverflow,
 paPrimingOutput
*}
type TPaStreamCallbackFlags = culong;

{** In a stream opened with paFramesPerBufferUnspecified, indicates that
 input data is all silence (zeros) because no real data is available. In a
 stream opened without paFramesPerBufferUnspecified, it indicates that one or
 more zero samples have been inserted into the input buffer to compensate
 for an input underflow.
 @see PaStreamCallbackFlags
*}
const paInputUnderflow   = TPaStreamCallbackFlags($00000001);

{** In a stream opened with paFramesPerBufferUnspecified, indicates that data
 prior to the first sample of the input buffer was discarded due to an
 overflow, possibly because the stream callback is using too much CPU time.
 Otherwise indicates that data prior to one or more samples in the
 input buffer was discarded.
 @see PaStreamCallbackFlags
*}
const paInputOverflow    = TPaStreamCallbackFlags($00000002);

{** Indicates that output data (or a gap) was inserted, possibly because the
 stream callback is using too much CPU time.
 @see PaStreamCallbackFlags
*}
const paOutputUnderflow  = TPaStreamCallbackFlags($00000004);

{** Indicates that output data will be discarded because no room is available.
 @see PaStreamCallbackFlags
*}
const paOutputOverflow   = TPaStreamCallbackFlags($00000008);

{** Some of all of the output data will be used to prime the stream, input
 data may be zero.
 @see PaStreamCallbackFlags
*}
const paPrimingOutput    = TPaStreamCallbackFlags($00000010);

{**
 Allowable return values for the PaStreamCallback.
 @see PaStreamCallback
*}
type TPaStreamCallbackResult = {enum}cInt; const
{enum_begin PaStreamCallbackResult}
    paContinue=0;
    paComplete=1;
    paAbort=2;
{enum_end PaStreamCallbackResult}

{**
 Functions of type PaStreamCallback are implemented by PortAudio clients.
 They consume, process or generate audio in response to requests from an
 active PortAudio stream.

 @param input and @param output are arrays of interleaved samples,
 the format, packing and number of channels used by the buffers are
 determined by parameters to Pa_OpenStream().

 @param frameCount The number of sample frames to be processed by
 the stream callback.

 @param timeInfo The time in seconds when the first sample of the input
 buffer was received at the audio input, the time in seconds when the first
 sample of the output buffer will begin being played at the audio output, and
 the time in seconds when the stream callback was called.
 See also Pa_GetStreamTime()

 @param statusFlags Flags indicating whether input and/or output buffers
 have been inserted or will be dropped to overcome underflow or overflow
 conditions.

 @param userData The value of a user supplied pointer passed to
 Pa_OpenStream() intended for storing synthesis data etc.

 @return
 The stream callback should return one of the values in the
 PaStreamCallbackResult enumeration. To ensure that the callback continues
 to be called, it should return paContinue (0). Either paComplete or paAbort
 can be returned to finish stream processing, after either of these values is
 returned the callback will not be called again. If paAbort is returned the
 stream will finish as soon as possible. If paComplete is returned, the stream
 will continue until all buffers generated by the callback have been played.
 This may be useful in applications such as soundfile players where a specific
 duration of output is required. However, it is not necessary to utilise this
 mechanism as Pa_StopStream(), Pa_AbortStream() or Pa_CloseStream() can also
 be used to stop the stream. The callback must always fill the entire output
 buffer irrespective of its return value.

 @see Pa_OpenStream, Pa_OpenDefaultStream

 @note With the exception of Pa_GetStreamCpuLoad() it is not permissable to call
 PortAudio API functions from within the stream callback.
*}
type
  PPaStreamCallback = ^TPaStreamCallback;
  TPaStreamCallback = function(
      input: Pointer; output: Pointer;
      frameCount: culong;
      timeInfo: PPaStreamCallbackTimeInfo;
      statusFlags: TPaStreamCallbackFlags;
      userData: Pointer ): cInt; cdecl;


{** Functions of type PaStreamFinishedCallback are implemented by PortAudio
 clients. They can be registered with a stream using the Pa_SetStreamFinishedCallback
 function. Once registered they are called when the stream becomes inactive
 (ie once a call to Pa_StopStream() will not block).
 A stream will become inactive after the stream callback returns non-zero,
 or when Pa_StopStream or Pa_AbortStream is called. For a stream providing audio
 output, if the stream callback returns paComplete, or Pa_StopStream is called,
 the stream finished callback will not be called until all generated sample data
 has been played.

 @param userData The userData parameter supplied to Pa_OpenStream()

 @see Pa_SetStreamFinishedCallback
*}
type
  PPaStreamFinishedCallback = ^TPaStreamFinishedCallback;
  TPaStreamFinishedCallback = procedure( userData: Pointer ); cdecl;


{** A structure containing unchanging information about an open stream.
 @see Pa_GetStreamInfo
*}
type
  PPaStreamInfo = ^TPaStreamInfo;
  TPaStreamInfo = record
      {** this is struct version 1 *}
      structVersion: cInt;

      {** The input latency of the stream in seconds. This value provides the most
       accurate estimate of input latency available to the implementation. It may
       differ significantly from the suggestedLatency value passed to Pa_OpenStream().
       The value of this field will be zero (0.) for output-only streams.
       @see PaTime
      *}
      inputLatency: TPaTime;

      {** The output latency of the stream in seconds. This value provides the most
       accurate estimate of output latency available to the implementation. It may
       differ significantly from the suggestedLatency value passed to Pa_OpenStream().
       The value of this field will be zero (0.) for input-only streams.
       @see PaTime
      *}
      outputLatency: TPaTime;

      {** The sample rate of the stream in Hertz (samples per second). In cases
       where the hardware sample rate is inaccurate and PortAudio is aware of it,
       the value of this field may be different from the sampleRate parameter
       passed to Pa_OpenStream(). If information about the actual hardware sample
       rate is not available, this field will have the same value as the sampleRate
       parameter passed to Pa_OpenStream().
      *}
      sampleRate: cdouble;
  end;

{** A special PaDeviceIndex value indicating that no device is available,
 or should be used.

 @see PaDeviceIndex
*}
const paNoDevice = TPaDeviceIndex(-1);


{** A special PaDeviceIndex value indicating that the device(s) to be used
 are specified in the host api specific stream info structure.

 @see PaDeviceIndex
*}
const paUseHostApiSpecificDeviceSpecification = TPaDeviceIndex(-2);




{$ifndef LOAD_PA_ON_RUNTIME}  // ("early binding")
{** Retrieve the release number of the currently running PortAudio build,
 For example, for version "19.5.1" this will return 0x00130501.
*}
function Pa_GetVersion(): cInt; cdecl; external LibName;


{** Retrieve a textual description of the current PortAudio build,
 eg "PortAudio V19-devel 13 October 2002".
*}
function Pa_GetVersionText(): pAnsiChar; cdecl; external LibName;


{** Translate the supplied PortAudio error code into a human readable
 message.
*}
function Pa_GetErrorText( errorCode: TPaError ): pAnsiChar; cdecl; external LibName;


{** Library initialization function - call this before using PortAudio.
 This function initialises internal data structures and prepares underlying
 host APIs for use.  With the exception of Pa_GetVersion(), Pa_GetVersionText(),
 and Pa_GetErrorText(), this function MUST be called before using any other
 PortAudio API functions.

 If Pa_Initialize() is called multiple times, each successful
 call must be matched with a corresponding call to Pa_Terminate(). 
 Pairs of calls to Pa_Initialize()/Pa_Terminate() may overlap, and are not 
 required to be fully nested.

 Note that if Pa_Initialize() returns an error code, Pa_Terminate() should
 NOT be called.

 @return paNoError if successful, otherwise an error code indicating the cause
 of failure.

 @see Pa_Terminate
*}
function Pa_Initialize(): TPaError; cdecl; external LibName;


{** Library termination function - call this when finished using PortAudio.
 This function deallocates all resources allocated by PortAudio since it was
 initializied by a call to Pa_Initialize(). In cases where Pa_Initialise() has
 been called multiple times, each call must be matched with a corresponding call
 to Pa_Terminate(). The final matching call to Pa_Terminate() will automatically
 close any PortAudio streams that are still open.

 Pa_Terminate() MUST be called before exiting a program which uses PortAudio.
 Failure to do so may result in serious resource leaks, such as audio devices
 not being available until the next reboot.

 @return paNoError if successful, otherwise an error code indicating the cause
 of failure.
 
 @see Pa_Initialize
*}
function Pa_Terminate(): TPaError; cdecl; external LibName;



{* Host API enumeration mechanism *}

{** Retrieve the number of available host APIs. Even if a host API is
 available it may have no devices available.

 @return A non-negative value indicating the number of available host APIs
 or, a PaErrorCode (which are always negative) if PortAudio is not initialized
 or an error is encountered.

 @see PaHostApiIndex
*}
function Pa_GetHostApiCount(): TPaHostApiIndex; cdecl; external LibName;


{** Retrieve the index of the default host API. The default host API will be
 the lowest common denominator host API on the current platform and is
 unlikely to provide the best performance.

 @return A non-negative value ranging from 0 to (Pa_GetHostApiCount()-1)
 indicating the default host API index or, a PaErrorCode (which are always
 negative) if PortAudio is not initialized or an error is encountered.
*}
function Pa_GetDefaultHostApi(): TPaHostApiIndex; cdecl; external LibName;



{** Retrieve a pointer to a structure containing information about a specific
 host Api.

 @param hostApi A valid host API index ranging from 0 to (Pa_GetHostApiCount()-1)

 @return A pointer to an immutable PaHostApiInfo structure describing
 a specific host API. If the hostApi parameter is out of range or an error
 is encountered, the function returns NULL.

 The returned structure is owned by the PortAudio implementation and must not
 be manipulated or freed. The pointer is only guaranteed to be valid between
 calls to Pa_Initialize() and Pa_Terminate().
*}
function Pa_GetHostApiInfo( hostApi: TPaHostApiIndex ): PPaHostApiInfo; cdecl; external LibName;


{** Convert a static host API unique identifier, into a runtime
 host API index.

 @param type A unique host API identifier belonging to the PaHostApiTypeId
 enumeration.

 @return A valid PaHostApiIndex ranging from 0 to (Pa_GetHostApiCount()-1) or,
 a PaErrorCode (which are always negative) if PortAudio is not initialized
 or an error is encountered.
 
 The paHostApiNotFound error code indicates that the host API specified by the
 type parameter is not available.

 @see PaHostApiTypeId
*}
function Pa_HostApiTypeIdToHostApiIndex( _type: TPaHostApiTypeId ): TPaHostApiIndex; cdecl; external LibName;


{** Convert a host-API-specific device index to standard PortAudio device index.
 This function may be used in conjunction with the deviceCount field of
 PaHostApiInfo to enumerate all devices for the specified host API.

 @param hostApi A valid host API index ranging from 0 to (Pa_GetHostApiCount()-1)

 @param hostApiDeviceIndex A valid per-host device index in the range
 0 to (Pa_GetHostApiInfo(hostApi)->deviceCount-1)

 @return A non-negative PaDeviceIndex ranging from 0 to (Pa_GetDeviceCount()-1)
 or, a PaErrorCode (which are always negative) if PortAudio is not initialized
 or an error is encountered.

 A paInvalidHostApi error code indicates that the host API index specified by
 the hostApi parameter is out of range.

 A paInvalidDevice error code indicates that the hostApiDeviceIndex parameter
 is out of range.
 
 @see PaHostApiInfo
*}
function Pa_HostApiDeviceIndexToDeviceIndex( hostApi: TPaHostApiIndex;
        hostApiDeviceIndex: cInt ): TPaDeviceIndex; cdecl; external LibName;



{** Return information about the last host error encountered. The error
 information returned by Pa_GetLastHostErrorInfo() will never be modified
 asyncronously by errors occurring in other PortAudio owned threads
 (such as the thread that manages the stream callback.)

 This function is provided as a last resort, primarily to enhance debugging
 by providing clients with access to all available error information.

 @return A pointer to an immutable structure constaining information about
 the host error. The values in this structure will only be valid if a
 PortAudio function has previously returned the paUnanticipatedHostError
 error code.
*}
function Pa_GetLastHostErrorInfo: PPaHostErrorInfo; cdecl; external LibName;



{* Device enumeration and capabilities *}

{** Retrieve the number of available devices. The number of available devices
 may be zero.

 @return A non-negative value indicating the number of available devices or,
 a PaErrorCode (which are always negative) if PortAudio is not initialized
 or an error is encountered.
*}
function Pa_GetDeviceCount(): TPaDeviceIndex; cdecl; external LibName;


{** Retrieve the index of the default input device. The result can be
 used in the inputDevice parameter to Pa_OpenStream().

 @return The default input device index for the default host API, or paNoDevice
 if no default input device is available or an error was encountered.
*}
function Pa_GetDefaultInputDevice(): TPaDeviceIndex; cdecl; external LibName;


{** Retrieve the index of the default output device. The result can be
 used in the outputDevice parameter to Pa_OpenStream().

 @return The default output device index for the defualt host API, or paNoDevice
 if no default output device is available or an error was encountered.

 @note
 On the PC, the user can specify a default device by
 setting an environment variable. For example, to use device #1.
<pre>
 set PA_RECOMMENDED_OUTPUT_DEVICE=1
</pre>
 The user should first determine the available device ids by using
 the supplied application "pa_devs".
*}
function Pa_GetDefaultOutputDevice(): TPaDeviceIndex; cdecl; external LibName;


{** Retrieve a pointer to a PaDeviceInfo structure containing information
 about the specified device.
 @return A pointer to an immutable PaDeviceInfo structure. If the device
 parameter is out of range the function returns NULL.

 @param device A valid device index in the range 0 to (Pa_GetDeviceCount()-1)

 @note PortAudio manages the memory referenced by the returned pointer,
 the client must not manipulate or free the memory. The pointer is only
 guaranteed to be valid between calls to Pa_Initialize() and Pa_Terminate().

 @see PaDeviceInfo, PaDeviceIndex
*}
function Pa_GetDeviceInfo( device: TPaDeviceIndex ): PPaDeviceInfo; cdecl; external LibName;


{** Return code for Pa_IsFormatSupported indicating success. *}
const paFormatIsSupported = (0);

{** Determine whether it would be possible to open a stream with the specified
 parameters.

 @param inputParameters A structure that describes the input parameters used to
 open a stream. The suggestedLatency field is ignored. See PaStreamParameters
 for a description of these parameters. inputParameters must be NULL for
 output-only streams.

 @param outputParameters A structure that describes the output parameters used
 to open a stream. The suggestedLatency field is ignored. See PaStreamParameters
 for a description of these parameters. outputParameters must be NULL for
 input-only streams.

 @param sampleRate The required sampleRate. For full-duplex streams it is the
 sample rate for both input and output

 @return Returns 0 if the format is supported, and an error code indicating why
 the format is not supported otherwise. The constant paFormatIsSupported is
 provided to compare with the return value for success.

 @see paFormatIsSupported, PaStreamParameters
*}
function Pa_IsFormatSupported( inputParameters: PPaStreamParameters;
                               outputParameters: PPaStreamParameters;
                               sampleRate: cdouble ): TPaError; cdecl; external LibName;



{* Streaming types and functions *}


{** Opens a stream for either input, output or both.
     
 @param stream The address of a PaStream pointer which will receive
 a pointer to the newly opened stream.
     
 @param inputParameters A structure that describes the input parameters used by
 the opened stream. See PaStreamParameters for a description of these parameters.
 inputParameters must be NULL for output-only streams.

 @param outputParameters A structure that describes the output parameters used by
 the opened stream. See PaStreamParameters for a description of these parameters.
 outputParameters must be NULL for input-only streams.
 
 @param sampleRate The desired sampleRate. For full-duplex streams it is the
 sample rate for both input and output

 @param framesPerBuffer The number of frames passed to the stream callback
 function, or the preferred block granularity for a blocking read/write stream.
 The special value paFramesPerBufferUnspecified (0) may be used to request that
 the stream callback will recieve an optimal (and possibly varying) number of
 frames based on host requirements and the requested latency settings.
 Note: With some host APIs, the use of non-zero framesPerBuffer for a callback
 stream may introduce an additional layer of buffering which could introduce
 additional latency. PortAudio guarantees that the additional latency
 will be kept to the theoretical minimum however, it is strongly recommended
 that a non-zero framesPerBuffer value only be used when your algorithm
 requires a fixed number of frames per stream callback.
 
 @param streamFlags Flags which modify the behaviour of the streaming process.
 This parameter may contain a combination of flags ORed together. Some flags may
 only be relevant to certain buffer formats.
     
 @param streamCallback A pointer to a client supplied function that is responsible
 for processing and filling input and output buffers. If this parameter is NULL
 the stream will be opened in 'blocking read/write' mode. In blocking mode,
 the client can receive sample data using Pa_ReadStream and write sample data
 using Pa_WriteStream, the number of samples that may be read or written
 without blocking is returned by Pa_GetStreamReadAvailable and
 Pa_GetStreamWriteAvailable respectively.

 @param userData A client supplied pointer which is passed to the stream callback
 function. It could for example, contain a pointer to instance data necessary
 for processing the audio buffers. This parameter is ignored if streamCallback
 is NULL.
     
 @return
 Upon success Pa_OpenStream() returns paNoError and places a pointer to a
 valid PaStream in the stream argument. The stream is inactive (stopped).
 If a call to Pa_OpenStream() fails, a non-zero error code is returned (see
 PaError for possible error codes) and the value of stream is invalid.

 @see PaStreamParameters, PaStreamCallback, Pa_ReadStream, Pa_WriteStream,
 Pa_GetStreamReadAvailable, Pa_GetStreamWriteAvailable
*}
function Pa_OpenStream( var stream: PPaStream;
                        inputParameters: PPaStreamParameters;
                        outputParameters: PPaStreamParameters;
                        sampleRate: cdouble;
                        framesPerBuffer: culong;
                        streamFlags: TPaStreamFlags;
                        streamCallback: PPaStreamCallback;
                        userData: Pointer ): TPaError; cdecl; external LibName;


{** A simplified version of Pa_OpenStream() that opens the default input
 and/or output devices.

 @param stream The address of a PaStream pointer which will receive
 a pointer to the newly opened stream.
 
 @param numInputChannels  The number of channels of sound that will be supplied
 to the stream callback or returned by Pa_ReadStream. It can range from 1 to
 the value of maxInputChannels in the PaDeviceInfo record for the default input
 device. If 0 the stream is opened as an output-only stream.

 @param numOutputChannels The number of channels of sound to be delivered to the
 stream callback or passed to Pa_WriteStream. It can range from 1 to the value
 of maxOutputChannels in the PaDeviceInfo record for the default output dvice.
 If 0 the stream is opened as an output-only stream.

 @param sampleFormat The sample format of both the input and output buffers
 provided to the callback or passed to and from Pa_ReadStream and Pa_WriteStream.
 sampleFormat may be any of the formats described by the PaSampleFormat
 enumeration.
 
 @param sampleRate Same as Pa_OpenStream parameter of the same name.
 @param framesPerBuffer Same as Pa_OpenStream parameter of the same name.
 @param streamCallback Same as Pa_OpenStream parameter of the same name.
 @param userData Same as Pa_OpenStream parameter of the same name.

 @return As for Pa_OpenStream

 @see Pa_OpenStream, PaStreamCallback
*}
function Pa_OpenDefaultStream( var stream: PPaStream;
                               numInputChannels: cInt;
                               numOutputChannels: cInt;
                               sampleFormat: TPaSampleFormat;
                               sampleRate: cdouble;
                               framesPerBuffer: culong;
                               streamCallback: PPaStreamCallback;
                               userData: Pointer ): TPaError; cdecl; external LibName;


{** Closes an audio stream. If the audio stream is active it
 discards any pending buffers as if Pa_AbortStream() had been called.
*}
function Pa_CloseStream( stream: PPaStream ): TPaError; cdecl; external LibName;


{** Register a stream finished callback function which will be called when the
 stream becomes inactive. See the description of PaStreamFinishedCallback for 
 further details about when the callback will be called.

 @param stream a pointer to a PaStream that is in the stopped state - if the
 stream is not stopped, the stream's finished callback will remain unchanged 
 and an error code will be returned.

 @param streamFinishedCallback a pointer to a function with the same signature
 as PaStreamFinishedCallback, that will be called when the stream becomes
 inactive. Passing NULL for this parameter will un-register a previously
 registered stream finished callback function.

 @return on success returns paNoError, otherwise an error code indicating the cause
 of the error.

 @see PaStreamFinishedCallback
*}
function Pa_SetStreamFinishedCallback( stream: PPaStream;
                streamFinishedCallback: PPaStreamFinishedCallback ): TPaError; cdecl; external LibName;


{** Commences audio processing.
*}
function Pa_StartStream( stream: PPaStream ): TPaError; cdecl; external LibName;


{** Terminates audio processing. It waits until all pending
 audio buffers have been played before it returns.
*}
function Pa_StopStream( stream: PPaStream ): TPaError; cdecl; external LibName;


{** Terminates audio processing immediately without waiting for pending
 buffers to complete.
*}
function Pa_AbortStream( stream: PPaStream ): TPaError; cdecl; external LibName;


{** Determine whether the stream is stopped.
 A stream is considered to be stopped prior to a successful call to
 Pa_StartStream and after a successful call to Pa_StopStream or Pa_AbortStream.
 If a stream callback returns a value other than paContinue the stream is NOT
 considered to be stopped.

 @return Returns one (1) when the stream is stopped, zero (0) when
 the stream is running or, a PaErrorCode (which are always negative) if
 PortAudio is not initialized or an error is encountered.

 @see Pa_StopStream, Pa_AbortStream, Pa_IsStreamActive
*}
function Pa_IsStreamStopped( stream: PPaStream ): TPaError; cdecl; external LibName;


{** Determine whether the stream is active.
 A stream is active after a successful call to Pa_StartStream(), until it
 becomes inactive either as a result of a call to Pa_StopStream() or
 Pa_AbortStream(), or as a result of a return value other than paContinue from
 the stream callback. In the latter case, the stream is considered inactive
 after the last buffer has finished playing.

 @return Returns one (1) when the stream is active (ie playing or recording
 audio), zero (0) when not playing or, a PaErrorCode (which are always negative)
 if PortAudio is not initialized or an error is encountered.

 @see Pa_StopStream, Pa_AbortStream, Pa_IsStreamStopped
*}
function Pa_IsStreamActive( stream: PPaStream ): TPaError; cdecl; external LibName;



{** Retrieve a pointer to a PaStreamInfo structure containing information
 about the specified stream.
 @return A pointer to an immutable PaStreamInfo structure. If the stream
 parameter invalid, or an error is encountered, the function returns NULL.

 @param stream A pointer to an open stream previously created with Pa_OpenStream.

 @note PortAudio manages the memory referenced by the returned pointer,
 the client must not manipulate or free the memory. The pointer is only
 guaranteed to be valid until the specified stream is closed.

 @see PaStreamInfo
*}
function Pa_GetStreamInfo( stream: PPaStream ): TPaStreamInfo; cdecl; external LibName;


{** Determine the current time for the stream according to the same clock used
 to generate buffer timestamps. This time may be used for syncronising other
 events to the audio stream, for example synchronizing audio to MIDI.
                                        
 @return The stream's current time in seconds, or 0 if an error occurred.

 @see PaTime, PaStreamCallback
*}
function Pa_GetStreamTime( stream: PPaStream ): TPaTime; cdecl; external LibName;


{** Retrieve CPU usage information for the specified stream.
 The "CPU Load" is a fraction of total CPU time consumed by a callback stream's
 audio processing routines including, but not limited to the client supplied
 stream callback. This function does not work with blocking read/write streams.

 This function may be called from the stream callback function or the
 application.
     
 @return
 A floating point value, typically between 0.0 and 1.0, where 1.0 indicates
 that the stream callback is consuming the maximum number of CPU cycles possible
 to maintain real-time operation. A value of 0.5 would imply that PortAudio and
 the stream callback was consuming roughly 50% of the available CPU time. The
 return value may exceed 1.0. A value of 0.0 will always be returned for a
 blocking read/write stream, or if an error occurrs.
*}
function Pa_GetStreamCpuLoad( stream: PPaStream ): cdouble; cdecl; external LibName;


{** Read samples from an input stream. The function doesn't return until
 the entire buffer has been filled - this may involve waiting for the operating
 system to supply the data.

 @param stream A pointer to an open stream previously created with Pa_OpenStream.
 
 @param buffer A pointer to a buffer of sample frames. The buffer contains
 samples in the format specified by the inputParameters->sampleFormat field
 used to open the stream, and the number of channels specified by
 inputParameters->numChannels. If non-interleaved samples were requested,
 buffer is a pointer to the first element of an array of non-interleaved
 buffer pointers, one for each channel.

 @param frames The number of frames to be read into buffer. This parameter
 is not constrained to a specific range, however high performance applications
 will want to match this parameter to the framesPerBuffer parameter used
 when opening the stream.

 @return On success PaNoError will be returned, or PaInputOverflowed if input
 data was discarded by PortAudio after the previous call and before this call.
*}
function Pa_ReadStream( stream: PPaStream;
                       buffer: Pointer;
                       frames: culong ): TPaError; cdecl; external LibName;


{** Write samples to an output stream. This function doesn't return until the
 entire buffer has been consumed - this may involve waiting for the operating
 system to consume the data.

 @param stream A pointer to an open stream previously created with Pa_OpenStream.

 @param buffer A pointer to a buffer of sample frames. The buffer contains
 samples in the format specified by the outputParameters->sampleFormat field
 used to open the stream, and the number of channels specified by
 outputParameters->numChannels. If non-interleaved samples were requested,
 buffer is a pointer to the first element of an array of non-interleaved
 buffer pointers, one for each channel.

 @param frames The number of frames to be written from buffer. This parameter
 is not constrained to a specific range, however high performance applications
 will want to match this parameter to the framesPerBuffer parameter used
 when opening the stream.

 @return On success PaNoError will be returned, or paOutputUnderflowed if
 additional output data was inserted after the previous call and before this
 call.
*}
function Pa_WriteStream( stream: PPaStream;
                        buffer: Pointer;
                        frames: culong ): TPaError; cdecl; external LibName;


{** Retrieve the number of frames that can be read from the stream without
 waiting.

 @return Returns a non-negative value representing the maximum number of frames
 that can be read from the stream without blocking or busy waiting or, a
 PaErrorCode (which are always negative) if PortAudio is not initialized or an
 error is encountered.
*}
function Pa_GetStreamReadAvailable( stream: PPaStream ): cslong; cdecl; external LibName;


{** Retrieve the number of frames that can be written to the stream without
 waiting.

 @return Returns a non-negative value representing the maximum number of frames
 that can be written to the stream without blocking or busy waiting or, a
 PaErrorCode (which are always negative) if PortAudio is not initialized or an
 error is encountered.
*}
function Pa_GetStreamWriteAvailable( stream: PPaStream ): cslong; cdecl; external LibName;


{** Retrieve the host type handling an open stream.

 @return Returns a non-negative value representing the host API type
 handling an open stream or, a PaErrorCode (which are always negative)
 if PortAudio is not initialized or an error is encountered.
*}
//function Pa_GetStreamHostApiType( stream: PPaStream ): TPaHostApiTypeId; cdecl; external LibName;
// --- commented out because, only used by Audacity,
// --- this is not an offical part of the Portaudio API !


{* Miscellaneous utilities *}

{** Retrieve the size of a given sample format in bytes.

 @return The size in bytes of a single sample in the specified format,
 or paSampleFormatNotSupported if the format is not supported.
*}
function Pa_GetSampleSize( format: TPaSampleFormat ): TPaError; cdecl; external LibName;


{** Put the caller to sleep for at least 'msec' milliseconds. This function is
 provided only as a convenience for authors of portable code (such as the tests
 and examples in the PortAudio distribution.)

 The function may sleep longer than requested so don't rely on this for accurate
 musical timing.
*}
procedure Pa_Sleep( msec: clong ); cdecl; external LibName;
{$endif not LOAD_PA_ON_RUNTIME}


{$ifdef LOAD_PA_ON_RUNTIME}  // ("late binding")
////////////////////////////////////////////////////////////////////////////////
// *************************** functions *******************************

var Pa_GetVersion: function():cInt ; cdecl;

var Pa_GetVersionText: function():pAnsiChar ; cdecl;

var Pa_GetErrorText: function(errorCode : TPaError):pAnsiChar ; cdecl;

var Pa_Initialize: function():TPaError ; cdecl;

var Pa_Terminate: function():TPaError ; cdecl;

var Pa_GetHostApiCount: function():TPaHostApiIndex; cdecl;

var Pa_GetDefaultHostApi: function():TPaHostApiIndex ; cdecl;

var Pa_GetHostApiInfo: function(hostApi : TPaHostApiIndex):PPaHostApiInfo; cdecl;

var Pa_HostApiTypeIdToHostApiIndex: function(_type : TPaHostApiTypeId):TPaHostApiIndex ; cdecl;

var Pa_HostApiDeviceIndexToDeviceIndex: function(hostApi : TPaHostApiIndex; hostApiDeviceIndex : cInt):TPaDeviceIndex ; cdecl;

var Pa_GetLastHostErrorInfo: function():PPaHostErrorInfo ; cdecl;


// ************** Device enumeration and capabilities ******************

var Pa_GetDeviceCount: function:TPaDeviceIndex ; cdecl;

var Pa_GetDefaultInputDevice: function:TPaDeviceIndex ; cdecl;

var Pa_GetDefaultOutputDevice: function:TPaDeviceIndex ; cdecl;

var Pa_GetDeviceInfo: function(device : TPaDeviceIndex):PPaDeviceInfo ; cdecl;

var Pa_IsFormatSupported: function(inputParameters,outputParameters : PPaStreamParameters; sampleRate : CDouble):TPaError ; cdecl;


// *********************** Stream function *****************************

var Pa_OpenStream: function( var stream : PPaStream;
                             inputParameters : PPaStreamParameters;
                             outputParameters : PPaStreamParameters;
                             sampleRate : CDouble;
                             framesPerBuffer : CULong;
                             streamFlags : TPaStreamFlags;
                             streamCallback : PPaStreamCallback;
                             userData : Pointer):TPaError ; cdecl;

var Pa_OpenDefaultStream: function( var stream : PPaStream;
                                    numInputChannels : cInt;
                                    numOutputChannels : cInt;
                                    sampleFormat : TPaSampleFormat;
                                    sampleRate : CDouble;
                                    framesPerBuffer : CULong;
                                    streamCallback : PPaStreamCallback;
                                    userData : Pointer):TPaError ; cdecl;

var Pa_CloseStream: function(stream : PPaStream):TPaError ; cdecl;

var Pa_SetStreamFinishedCallback: function( stream : PPaStream;
                                            streamFinishedCallback : PPaStreamFinishedCallback):TPaError ; cdecl;

var Pa_StartStream: function(stream : PPaStream):TPaError ; cdecl;

var Pa_StopStream: function(stream : PPaStream):TPaError ; cdecl;

var Pa_AbortStream: function(stream : PPaStream):TPaError ; cdecl;

var Pa_IsStreamStopped: function(stream : PPaStream):TPaError ; cdecl;

var Pa_IsStreamActive: function(stream : PPaStream):TPaError ; cdecl;

var Pa_GetStreamInfo: function(stream : PPaStream):PPaStreamInfo ; cdecl;

var Pa_GetStreamTime: function(stream : PPaStream):TPaTime ; cdecl;

var Pa_GetStreamCpuLoad: function(stream : PPaStream):CDouble ; cdecl;

var Pa_ReadStream: function(stream : PPaStream; buffer : Pointer;frames : CULong):TPaError ; cdecl;

var Pa_WriteStream: function(stream : PPaStream; buffer : Pointer;frames : CULong):TPaError ; cdecl;

var Pa_GetStreamReadAvailable: function(stream : PPaStream):CSLong ; cdecl;

var Pa_GetStreamWriteAvailable: function(stream : PPaStream):CSLong ; cdecl;


// ****************** Miscellaneous utilities **************************

var Pa_GetSampleSize: function(format : TPaSampleFormat):TPaError ; cdecl;

var Pa_Sleep: function(msec : CLong) : integer; cdecl;


// functions for dynamic loading the Portaudio DLL

var Pa_Handle : TLibHandle=dynlibs.NilHandle; // this will hold our handle for the lib;
                                              // it functions nicely as a multi-lib prevention unit
                                              // as well...
var ReferenceCounter : cardinal = 0;  // Reference counter


function Pa_IsLoaded : boolean; inline;

function Pa_Load( const libfilename:string):boolean; // load the lib

procedure Pa_Unload(); // unload and free the lib from memory :
                       // do not forget to call it
                       // before closing Your application !
////////////////////////////////////////////////////////////////////////////////
{$endif LOAD_PA_ON_RUNTIME}


implementation


{$ifdef LOAD_PA_ON_RUNTIME}
function Pa_IsLoaded: boolean;
begin
  Result := (Pa_Handle <> dynlibs.NilHandle);
end;

function Pa_Load( const libfilename:string):boolean;
begin
  Result := False;
  if Pa_Handle <> 0 
  then 
    begin
	  Inc(ReferenceCounter);
	  Result:= TRUE {is it already there ?}
	end  
  else 
    begin {go & load the library}
      if Length( libfilename) = 0 then exit;
      Pa_Handle:= DynLibs.LoadLibrary( libfilename); // obtain the handle we want
      if Pa_Handle <> DynLibs.NilHandle then
        begin
          {now we tie the functions to the VARs from above}
          Pointer( Pa_GetVersion):= DynLibs.GetProcedureAddress( PA_Handle, pAnsiChar( 'Pa_GetVersion'));
          Pointer( Pa_GetVersionText):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetVersionText'));
          Pointer( Pa_GetErrorText):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetErrorText'));
          Pointer( Pa_Initialize):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_Initialize'));
          Pointer( Pa_Terminate):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_Terminate'));
          Pointer( Pa_GetHostApiCount):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetHostApiCount'));
          Pointer( Pa_GetDefaultHostApi):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetDefaultHostApi'));
          Pointer( Pa_GetHostApiInfo):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetHostApiInfo'));
          Pointer( Pa_HostApiTypeIdToHostApiIndex):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_HostApiTypeIdToHostApiIndex'));
          Pointer( Pa_HostApiDeviceIndexToDeviceIndex):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_HostApiDeviceIndexToDeviceIndex'));
          Pointer( Pa_GetLastHostErrorInfo):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetLastHostErrorInfo'));
          //////////////////
          Pointer( Pa_GetDeviceCount):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetDeviceCount'));
          Pointer( Pa_GetDefaultInputDevice):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetDefaultInputDevice'));
          Pointer( Pa_GetDefaultOutputDevice):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetDefaultOutputDevice'));
          Pointer( Pa_GetDeviceInfo):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetDeviceInfo'));
          Pointer( Pa_IsFormatSupported):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_IsFormatSupported'));
          //////////////////////
          Pointer( Pa_OpenStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_OpenStream'));
          Pointer( Pa_OpenDefaultStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_OpenDefaultStream'));
          Pointer( Pa_CloseStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_CloseStream'));
          Pointer( Pa_SetStreamFinishedCallback):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_SetStreamFinishedCallback'));
          Pointer( Pa_StartStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_StartStream'));
          Pointer( Pa_StopStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_StopStream'));
          Pointer( Pa_AbortStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_AbortStream'));
          Pointer( Pa_IsStreamStopped):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_IsStreamStopped'));
          Pointer( Pa_IsStreamActive):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_IsStreamActive'));
          Pointer( Pa_GetStreamInfo):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetStreamInfo'));
          Pointer( Pa_GetStreamTime):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetStreamTime'));
          Pointer( Pa_GetStreamCpuLoad):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetStreamCpuLoad'));
          Pointer( Pa_ReadStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_ReadStream'));
          Pointer( Pa_WriteStream):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_WriteStream'));
          Pointer( Pa_GetStreamReadAvailable):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetStreamReadAvailable'));
          Pointer( Pa_GetStreamWriteAvailable):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetStreamWriteAvailable'));
          Pointer( Pa_GetSampleSize):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_GetSampleSize'));
          Pointer( Pa_Sleep):=DynLibs.GetProcedureAddress( PA_Handle,pAnsiChar('Pa_Sleep'));
        end;
      Result := Pa_IsLoaded;
      ReferenceCounter:=1;
    end;
end;

procedure Pa_Unload;
begin
  // < Reference counting
  if ReferenceCounter > 0 then
    dec(ReferenceCounter);
  if ReferenceCounter > 0 then
    exit;
  // >
  if Pa_IsLoaded then
  begin
    Pa_Terminate();
    DynLibs.UnloadLibrary( Pa_Handle);
    Pa_Handle:= DynLibs.NilHandle;
  end;
end;
{$endif LOAD_PA_ON_RUNTIME}


end.
